/* Copyright (C) 2019-2022 Greenbone AG
 *
 * SPDX-License-Identifier: AGPL-3.0-or-later
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

import {
  parseCvssV2BaseFromVector,
  parseCvssV2BaseVector,
  parseCvssV3BaseFromVector,
  parseCvssV3BaseVector,
} from '../parser/cvss';

const cvss_v2_entries = {
  entries: [
    {vector: 'AV:A/AC:H/Au:M/C:N/I:P/A:P', value: 2.7},
    {vector: 'AV:A/AC:H/Au:N/C:C/I:C/A:C', value: 6.8},
    {vector: 'AV:A/AC:H/Au:N/C:C/I:N/A:C', value: 6.2},
    {vector: 'AV:A/AC:H/Au:N/C:C/I:P/A:N', value: 5.3},
    {vector: 'AV:A/AC:H/Au:N/C:N/I:N/A:C', value: 4.6},
    {vector: 'AV:A/AC:H/Au:N/C:N/I:N/A:P', value: 1.8},
    {vector: 'AV:A/AC:H/Au:N/C:N/I:P/A:N', value: 1.8},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:N/A:N', value: 1.8},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:N/A:P', value: 3.2},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:P/A:C', value: 5.8},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:P/A:N', value: 3.2},
    {vector: 'AV:A/AC:H/Au:N/C:P/I:P/A:P', value: 4.3},
    {vector: 'AV:A/AC:H/Au:S/C:C/I:C/A:C', value: 6.5},
    {vector: 'AV:A/AC:H/Au:S/C:C/I:N/A:N', value: 4.3},
    {vector: 'AV:A/AC:H/Au:S/C:N/I:N/A:C', value: 4.3},
    {vector: 'AV:A/AC:H/Au:S/C:N/I:N/A:P', value: 1.4},
    {vector: 'AV:A/AC:H/Au:S/C:P/I:N/A:P', value: 2.9},
    {vector: 'AV:A/AC:H/Au:S/C:P/I:P/A:C', value: 5.5},
    {vector: 'AV:A/AC:H/Au:S/C:P/I:P/A:P', value: 4.0},
    {vector: 'AV:A/AC:L/Au:M/C:C/I:C/A:C', value: 7.2},
    {vector: 'AV:A/AC:L/Au:M/C:P/I:P/A:P', value: 4.7},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:C/A:C', value: 8.3},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:C/A:N', value: 7.8},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:N/A:N', value: 6.1},
    {vector: 'AV:A/AC:L/Au:N/C:C/I:P/A:C', value: 8.0},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:C/A:C', value: 7.8},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:C/A:N', value: 6.1},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:N/A:C', value: 6.1},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:N/A:P', value: 3.3},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:P/A:C', value: 6.8},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:P/A:N', value: 3.3},
    {vector: 'AV:A/AC:L/Au:N/C:N/I:P/A:P', value: 4.8},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:C/A:C', value: 8.0},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:N/A:N', value: 3.3},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:N/A:P', value: 4.8},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:P/A:C', value: 7.3},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:P/A:N', value: 4.8},
    {vector: 'AV:A/AC:L/Au:N/C:P/I:P/A:P', value: 5.8},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:C/A:C', value: 7.7},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:C/A:N', value: 7.1},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:C/A:P', value: 7.4},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:N/A:N', value: 5.5},
    {vector: 'AV:A/AC:L/Au:S/C:C/I:P/A:P', value: 6.7},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:N/A:C', value: 5.5},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:N/A:P', value: 2.7},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:P/A:C', value: 6.2},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:P/A:N', value: 2.7},
    {vector: 'AV:A/AC:L/Au:S/C:N/I:P/A:P', value: 4.1},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:N/A:N', value: 2.7},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:N/A:P', value: 4.1},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:P/A:C', value: 6.7},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:P/A:N', value: 4.1},
    {vector: 'AV:A/AC:L/Au:S/C:P/I:P/A:P', value: 5.2},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:C/A:C', value: 7.9},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:C/A:N', value: 7.3},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:C/A:P', value: 7.6},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:N/A:C', value: 7.3},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:N/A:N', value: 5.7},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:P/A:C', value: 7.6},
    {vector: 'AV:A/AC:M/Au:N/C:C/I:P/A:P', value: 6.9},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:C/A:C', value: 7.3},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:C/A:N', value: 5.7},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:N/A:C', value: 5.7},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:N/A:P', value: 2.9},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:P/A:C', value: 6.4},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:P/A:N', value: 2.9},
    {vector: 'AV:A/AC:M/Au:N/C:N/I:P/A:P', value: 4.3},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:C/A:C', value: 7.6},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:N/A:N', value: 2.9},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:N/A:P', value: 4.3},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:P/A:N', value: 4.3},
    {vector: 'AV:A/AC:M/Au:N/C:P/I:P/A:P', value: 5.4},
    {vector: 'AV:A/AC:M/Au:S/C:C/I:C/A:C', value: 7.4},
    {vector: 'AV:A/AC:M/Au:S/C:C/I:N/A:N', value: 5.2},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:N/A:C', value: 5.2},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:N/A:P', value: 2.3},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:P/A:C', value: 5.8},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:P/A:N', value: 2.3},
    {vector: 'AV:A/AC:M/Au:S/C:N/I:P/A:P', value: 3.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:N/A:C', value: 5.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:N/A:N', value: 2.3},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:N/A:P', value: 3.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:P/A:C', value: 6.3},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:P/A:N', value: 3.8},
    {vector: 'AV:A/AC:M/Au:S/C:P/I:P/A:P', value: 4.9},
    {vector: 'AV:L/AC:H/Au:M/C:C/I:C/A:C', value: 5.9},
    {vector: 'AV:L/AC:H/Au:M/C:N/I:N/A:C', value: 3.7},
    {vector: 'AV:L/AC:H/Au:M/C:P/I:P/A:C', value: 4.9},
    {vector: 'AV:L/AC:H/Au:M/C:P/I:P/A:P', value: 3.4},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:C/A:C', value: 6.2},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:C/A:N', value: 5.6},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:C/A:P', value: 5.9},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:N/A:C', value: 5.6},
    {vector: 'AV:L/AC:H/Au:N/C:C/I:N/A:N', value: 4.0},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:C/A:C', value: 5.6},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:C/A:N', value: 4.0},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:N/A:C', value: 4.0},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:N/A:P', value: 1.2},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:P/A:N', value: 1.2},
    {vector: 'AV:L/AC:H/Au:N/C:N/I:P/A:P', value: 2.6},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:N/A:C', value: 4.7},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:N/A:N', value: 1.2},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:N/A:P', value: 2.6},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:P/A:C', value: 5.2},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:P/A:N', value: 2.6},
    {vector: 'AV:L/AC:H/Au:N/C:P/I:P/A:P', value: 3.7},
    {vector: 'AV:L/AC:H/Au:S/C:C/I:C/A:C', value: 6.0},
    {vector: 'AV:L/AC:H/Au:S/C:C/I:C/A:N', value: 5.5},
    {vector: 'AV:L/AC:H/Au:S/C:C/I:N/A:N', value: 3.8},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:C/A:C', value: 5.5},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:C/A:N', value: 3.8},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:N/A:C', value: 3.8},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:N/A:P', value: 1.0},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:P/A:N', value: 1.0},
    {vector: 'AV:L/AC:H/Au:S/C:N/I:P/A:P', value: 2.4},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:N/A:N', value: 1.0},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:N/A:P', value: 2.4},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:P/A:N', value: 2.4},
    {vector: 'AV:L/AC:H/Au:S/C:P/I:P/A:P', value: 3.5},
    {vector: 'AV:L/AC:L/Au:M/C:C/I:C/A:C', value: 6.5},
    {vector: 'AV:L/AC:L/Au:M/C:N/I:N/A:C', value: 4.3},
    {vector: 'AV:L/AC:L/Au:M/C:P/I:N/A:N', value: 1.4},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:C/A:C', value: 7.2},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:C/A:N', value: 6.6},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:C/A:P', value: 6.8},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:N/A:C', value: 6.6},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:N/A:N', value: 4.9},
    {vector: 'AV:L/AC:L/Au:N/C:C/I:P/A:P', value: 6.1},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:C/A:C', value: 6.6},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:C/A:N', value: 4.9},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:N/A:C', value: 4.9},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:N/A:N', value: 0.0},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:N/A:P', value: 2.1},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:P/A:C', value: 5.6},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:P/A:N', value: 2.1},
    {vector: 'AV:L/AC:L/Au:N/C:N/I:P/A:P', value: 3.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:C/A:C', value: 6.8},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:C/A:N', value: 5.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:C/A:P', value: 6.1},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:N/A:C', value: 5.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:N/A:N', value: 2.1},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:N/A:P', value: 3.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:P/A:C', value: 6.1},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:P/A:N', value: 3.6},
    {vector: 'AV:L/AC:L/Au:N/C:P/I:P/A:P', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:C/A:C', value: 6.8},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:C/A:N', value: 6.2},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:N/A:N', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:C/I:N/A:P', value: 5.2},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:C/A:C', value: 6.2},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:C/A:N', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:N/A:C', value: 4.6},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:N/A:P', value: 1.7},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:P/A:C', value: 5.2},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:P/A:N', value: 1.7},
    {vector: 'AV:L/AC:L/Au:S/C:N/I:P/A:P', value: 3.2},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:N/A:N', value: 1.7},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:N/A:P', value: 3.2},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:P/A:C', value: 5.7},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:P/A:N', value: 3.2},
    {vector: 'AV:L/AC:L/Au:S/C:P/I:P/A:P', value: 4.3},
    {vector: 'AV:L/AC:M/Au:M/C:C/I:C/A:C', value: 6.3},
    {vector: 'AV:L/AC:M/Au:M/C:N/I:N/A:C', value: 4.1},
    {vector: 'AV:L/AC:M/Au:M/C:N/I:P/A:P', value: 2.7},
    {vector: 'AV:L/AC:M/Au:M/C:P/I:N/A:N', value: 1.3},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:C/A:C', value: 6.9},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:C/A:N', value: 6.3},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:C/A:P', value: 6.6},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:N/A:C', value: 6.3},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:N/A:N', value: 4.7},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:N/A:P', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:P/A:C', value: 6.6},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:P/A:N', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:C/I:P/A:P', value: 5.9},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:C/A:C', value: 6.3},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:C/A:N', value: 4.7},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:N/A:C', value: 4.7},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:N/A:P', value: 1.9},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:P/A:C', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:P/A:N', value: 1.9},
    {vector: 'AV:L/AC:M/Au:N/C:N/I:P/A:P', value: 3.3},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:C/A:C', value: 6.6},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:C/A:N', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:N/A:C', value: 5.4},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:N/A:N', value: 1.9},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:N/A:P', value: 3.3},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:P/A:C', value: 5.9},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:P/A:N', value: 3.3},
    {vector: 'AV:L/AC:M/Au:N/C:P/I:P/A:P', value: 4.4},
    {vector: 'AV:L/AC:M/Au:S/C:C/I:C/A:C', value: 6.6},
    {vector: 'AV:L/AC:M/Au:S/C:C/I:C/A:N', value: 6.0},
    {vector: 'AV:L/AC:M/Au:S/C:C/I:N/A:N', value: 4.4},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:C/A:C', value: 6.0},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:N/A:C', value: 4.4},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:N/A:P', value: 1.5},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:P/A:N', value: 1.5},
    {vector: 'AV:L/AC:M/Au:S/C:N/I:P/A:P', value: 3.0},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:N/A:N', value: 1.5},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:P/A:C', value: 5.5},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:P/A:N', value: 3.0},
    {vector: 'AV:L/AC:M/Au:S/C:P/I:P/A:P', value: 4.1},
    {vector: 'AV:N/AC:H/Au:M/C:C/I:C/A:C', value: 6.8},
    {vector: 'AV:N/AC:H/Au:M/C:N/I:N/A:P', value: 1.7},
    {vector: 'AV:N/AC:H/Au:M/C:N/I:P/A:N', value: 1.7},
    {vector: 'AV:N/AC:H/Au:M/C:P/I:N/A:N', value: 1.7},
    {vector: 'AV:N/AC:H/Au:M/C:P/I:P/A:N', value: 3.2},
    {vector: 'AV:N/AC:H/Au:M/C:P/I:P/A:P', value: 4.3},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:C/A:C', value: 7.6},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:C/A:N', value: 7.1},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:C/A:P', value: 7.3},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:N/A:C', value: 7.1},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:N/A:N', value: 5.4},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:P/A:C', value: 7.3},
    {vector: 'AV:N/AC:H/Au:N/C:C/I:P/A:N', value: 6.1},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:C/A:C', value: 7.1},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:C/A:N', value: 5.4},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:N/A:C', value: 5.4},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:N/A:P', value: 2.6},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:P/A:C', value: 6.1},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:P/A:N', value: 2.6},
    {vector: 'AV:N/AC:H/Au:N/C:N/I:P/A:P', value: 4.0},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:N/A:N', value: 2.6},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:N/A:P', value: 4.0},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:P/A:C', value: 6.6},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:P/A:N', value: 4.0},
    {vector: 'AV:N/AC:H/Au:N/C:P/I:P/A:P', value: 5.1},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:C/A:C', value: 7.1},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:C/A:N', value: 6.6},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:C/A:P', value: 6.8},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:N/A:C', value: 6.6},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:N/A:N', value: 4.9},
    {vector: 'AV:N/AC:H/Au:S/C:C/I:P/A:P', value: 6.1},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:C/A:C', value: 6.6},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:N/A:C', value: 4.9},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:N/A:P', value: 2.1},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:P/A:C', value: 5.6},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:P/A:N', value: 2.1},
    {vector: 'AV:N/AC:H/Au:S/C:N/I:P/A:P', value: 3.6},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:N/A:N', value: 2.1},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:N/A:P', value: 3.6},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:P/A:C', value: 6.1},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:P/A:N', value: 3.6},
    {vector: 'AV:N/AC:H/Au:S/C:P/I:P/A:P', value: 4.6},
    {vector: 'AV:N/AC:L/Au:M/C:C/I:C/A:C', value: 8.3},
    {vector: 'AV:N/AC:L/Au:M/C:C/I:C/A:N', value: 7.7},
    {vector: 'AV:N/AC:L/Au:M/C:N/I:N/A:C', value: 6.1},
    {vector: 'AV:N/AC:L/Au:M/C:N/I:N/A:P', value: 3.3},
    {vector: 'AV:N/AC:L/Au:M/C:N/I:P/A:N', value: 3.3},
    {vector: 'AV:N/AC:L/Au:M/C:P/I:P/A:N', value: 4.7},
    {vector: 'AV:N/AC:L/Au:M/C:P/I:P/A:P', value: 5.8},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:C/A:C', value: 10.0},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:C/A:N', value: 9.4},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:C/A:P', value: 9.7},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:N/A:C', value: 9.4},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:N/A:N', value: 7.8},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:N/A:P', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:P/A:N', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:C/I:P/A:P', value: 9.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:C/A:C', value: 9.4},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:C/A:N', value: 7.8},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:C/A:P', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:N/A:C', value: 7.8},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:N/A:N', value: 0.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:N/A:P', value: 5.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:P/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:P/A:N', value: 5.0},
    {vector: 'AV:N/AC:L/Au:N/C:N/I:P/A:P', value: 6.4},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:C/A:C', value: 9.7},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:C/A:N', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:C/A:P', value: 9.0},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:N/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:N/A:N', value: 5.0},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:N/A:P', value: 6.4},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:P/A:C', value: 9.0},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:P/A:N', value: 6.4},
    {vector: 'AV:N/AC:L/Au:N/C:P/I:P/A:P', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:C/A:C', value: 9.0},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:C/A:N', value: 8.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:C/A:P', value: 8.7},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:N/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:N/A:N', value: 6.8},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:N/A:P', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:P/A:C', value: 8.7},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:P/A:N', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:C/I:P/A:P', value: 8.0},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:C/A:C', value: 8.5},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:C/A:N', value: 6.8},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:C/A:P', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:N/A:C', value: 6.8},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:N/A:P', value: 4.0},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:P/A:C', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:P/A:N', value: 4.0},
    {vector: 'AV:N/AC:L/Au:S/C:N/I:P/A:P', value: 5.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:C/A:C', value: 8.7},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:C/A:N', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:C/A:P', value: 8.0},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:N/A:C', value: 7.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:N/A:N', value: 4.0},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:N/A:P', value: 5.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:P/A:C', value: 8.0},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:P/A:N', value: 5.5},
    {vector: 'AV:N/AC:L/Au:S/C:P/I:P/A:P', value: 6.5},
    {vector: 'AV:N/AC:M/Au:M/C:C/I:C/A:C', value: 7.9},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:N/A:C', value: 5.7},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:N/A:P', value: 2.8},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:P/A:N', value: 2.8},
    {vector: 'AV:N/AC:M/Au:M/C:N/I:P/A:P', value: 4.3},
    {vector: 'AV:N/AC:M/Au:M/C:P/I:N/A:N', value: 2.8},
    {vector: 'AV:N/AC:M/Au:M/C:P/I:P/A:N', value: 4.3},
    {vector: 'AV:N/AC:M/Au:M/C:P/I:P/A:P', value: 5.4},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:C/A:C', value: 9.3},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:C/A:N', value: 8.8},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:N/A:N', value: 7.1},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:P/A:C', value: 9.0},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:P/A:N', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:C/I:P/A:P', value: 8.3},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:C/A:C', value: 8.8},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:C/A:N', value: 7.1},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:N/A:C', value: 7.1},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:N/A:P', value: 4.3},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:P/A:C', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:P/A:N', value: 4.3},
    {vector: 'AV:N/AC:M/Au:N/C:N/I:P/A:P', value: 5.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:C/A:C', value: 9.0},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:C/A:N', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:C/A:P', value: 8.3},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:N/A:C', value: 7.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:N/A:N', value: 4.3},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:N/A:P', value: 5.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:P/A:C', value: 8.3},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:P/A:N', value: 5.8},
    {vector: 'AV:N/AC:M/Au:N/C:P/I:P/A:P', value: 6.8},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:C/A:C', value: 8.5},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:C/A:N', value: 7.9},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:C/A:P', value: 8.2},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:N/A:C', value: 7.9},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:N/A:N', value: 6.3},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:P/A:N', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:C/I:P/A:P', value: 7.5},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:C/A:C', value: 7.9},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:C/A:N', value: 6.3},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:N/A:C', value: 6.3},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:N/A:N', value: 0.0},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:N/A:P', value: 3.5},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:P/A:C', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:P/A:N', value: 3.5},
    {vector: 'AV:N/AC:M/Au:S/C:N/I:P/A:P', value: 4.9},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:C/A:C', value: 8.2},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:C/A:N', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:N/A:C', value: 7.0},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:N/A:N', value: 3.5},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:N/A:P', value: 4.9},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:P/A:C', value: 7.5},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:P/A:N', value: 4.9},
    {vector: 'AV:N/AC:M/Au:S/C:P/I:P/A:P', value: 6.0},
  ],
};

describe('CVSS v2 parser tests', () => {
  test('should return correct severity value for cvss v2 vector', () => {
    cvss_v2_entries.entries.forEach(element => {
      const {
        accessVector,
        accessComplexity,
        authentication,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
        cvssScore,
      } = parseCvssV2BaseFromVector(element.vector);
      expect(cvssScore).toEqual(element.value);
      const cvssV2 = parseCvssV2BaseVector({
        accessVector,
        accessComplexity,
        authentication,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
      });
      expect(cvssV2[0]).toEqual(element.vector);
      expect(cvssV2[1]).toEqual(element.value);
    });
  });
});

const cvss_v3_entries = {
  entries: [
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:H/I:L/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 2.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:L/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 6.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 5.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:L/A:L', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 1.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 1.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:H/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:N/I:N/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:L/I:L/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 1.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:N/A:L', value: 1.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:L/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:H/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 4.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:H/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 2.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 4.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 2.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 2.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:H', value: 5.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 4.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:N', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:H/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:L/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:L/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:L/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 8.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 3.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:L', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:N/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:N', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:H/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:L', value: 7.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:L', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:H/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:N/A:N', value: 2.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:L/A:L', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:N/A:L', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 3.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:H/A:H', value: 8.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:L/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:H/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:N/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:L', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:L/A:N', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 6.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:L/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:H/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:N/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:L/A:L', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:L', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:N', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:H/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:L', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:L/A:L', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:L/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:L', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:L', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 9.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:N/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 6.9},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:L', value: 6.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 6.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:C/C:H/I:L/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:L/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 3.7},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 2.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 2.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 2.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:C/C:H/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:A/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 5.2},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:A/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:C/C:N/I:N/A:L', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:C/C:N/I:H/A:H', value: 6.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:H/UI:R/S:U/C:N/I:N/A:L', value: 1.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 5.8},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:L', value: 7.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 7.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 2.5},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:N/I:H/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:L/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:L/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:L/A:N', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 3.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:L', value: 3.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:N', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:H', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:C/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:L/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:H/UI:R/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:H', value: 7.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:L', value: 3.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:H/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 3.9},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 2.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 9.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 3.6},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.8},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:L', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.3},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:L/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:N/I:H/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:C/C:N/I:L/A:N', value: 3.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:L', value: 5.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:C/C:L/I:N/A:N', value: 2.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 2.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 4.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 3.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:H/A:L', value: 6.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:L/I:H/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:L/A:L', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:L/I:N/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:H', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:L', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 4.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 7.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 3.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:N', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:H/I:L/A:L', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 3.4},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:H/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 3.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:L/A:N', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:C/C:N/I:N/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:L/A:L', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:H/A:H', value: 6.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:H', value: 6.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:L', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:L/A:N', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:L/I:N/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:H', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:L', value: 3.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:L/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 4.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:L', value: 2.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:H/A:H', value: 8.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:L/A:N', value: 6.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:H/I:N/A:N', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:L/I:L/A:N', value: 4.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:C/C:N/I:N/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:H/I:N/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:H/A:N', value: 4.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:H/UI:R/S:U/C:N/I:L/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:L', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:L/A:N', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:H/I:N/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:H/A:N', value: 8.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:L', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:L/A:N', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:L/I:N/A:N', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:H/A:N', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:L/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:H', value: 7.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:C/C:N/I:N/A:L', value: 5.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:H', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:H', value: 9.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:L', value: 8.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:H/A:N', value: 8.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:L/A:N', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:H/I:N/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:L/I:N/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:H/A:N', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:C/C:N/I:L/A:N', value: 4.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:H', value: 8.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:L', value: 7.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:L/A:N', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:H/I:N/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:H/A:L', value: 6.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:L', value: 5.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:L/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:L/I:N/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:L', value: 4.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:H', value: 5.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:L/UI:R/S:U/C:N/I:N/A:L', value: 3.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:N', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:L/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:L', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:N/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:H', value: 9.9},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:L/A:N', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:L', value: 7.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:L/I:N/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:H', value: 10.0},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:L/A:N', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:N/I:N/A:L', value: 5.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 9.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:L', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:H', value: 9.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:L', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:H', value: 8.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 7.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 9.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:L', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 7.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 5.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:H', value: 9.6},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:H/A:N', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:L/A:L', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:L/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:H/I:N/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:H/A:N', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:L/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:L/I:N/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:H', value: 9.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:H/A:N', value: 7.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:H', value: 8.2},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:C/C:N/I:L/A:N', value: 4.7},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 8.8},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:L', value: 8.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:N', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:L/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:L', value: 6.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:L/A:N', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:L/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:H', value: 8.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:L', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:L', value: 5.4},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:L/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:H', value: 6.5},
    {vector: 'CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:N/I:N/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:U/C:H/I:H/A:L', value: 5.6},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:H/UI:R/S:U/C:L/I:L/A:N', value: 2.7},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.1},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.2},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:R/S:C/C:H/I:H/A:L', value: 6.9},
    {vector: 'CVSS:3.1/AV:P/AC:H/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:C/C:H/I:H/A:H', value: 7.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:H/A:H', value: 6.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:H/I:N/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:H/A:N', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:H/UI:N/S:U/C:N/I:N/A:H', value: 3.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:C/C:H/I:H/A:N', value: 7.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:L', value: 6.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:H/A:N', value: 5.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:L/A:N', value: 4.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:H', value: 5.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:L', value: 4.9},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:H', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:N/S:U/C:N/I:N/A:L', value: 2.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:L/UI:R/S:U/C:H/I:H/A:N', value: 5.7},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H', value: 7.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:C/C:N/I:H/A:N', value: 5.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H', value: 6.8},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:L', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:N', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:L/A:H', value: 6.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:L', value: 5.2},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:H/I:N/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:L', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:L/A:N', value: 3.5},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:L/I:N/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:H', value: 6.1},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:H/A:N', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:L/A:N', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:H', value: 4.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:N/S:U/C:N/I:N/A:L', value: 2.4},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H', value: 6.6},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:H/I:N/A:N', value: 4.3},
    {vector: 'CVSS:3.1/AV:P/AC:L/PR:N/UI:R/S:U/C:N/I:H/A:N', value: 4.3},
  ],
};

describe('CVSS v3 parser tests', () => {
  test('should return correct severity value for cvss v3 vector', () => {
    cvss_v3_entries.entries.forEach(element => {
      const {
        attackVector,
        attackComplexity,
        privilegesRequired,
        userInteraction,
        scope,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
        cvssScore,
      } = parseCvssV3BaseFromVector(element.vector);
      expect(cvssScore).toEqual(element.value);
      const cvssV3 = parseCvssV3BaseVector({
        attackVector,
        attackComplexity,
        privilegesRequired,
        userInteraction,
        scope,
        confidentialityImpact,
        availabilityImpact,
        integrityImpact,
      });
      expect(cvssV3[0]).toEqual(element.vector);
      expect(cvssV3[1]).toEqual(element.value);
    });
  });
});
