#!/usr/bin/env python
# coding: utf-8

"""
Converts a native Python dictionary into an XML string. Supports int, float, str, unicode, list, dict and arbitrary nesting.
"""

from __future__ import unicode_literals

__version__ = '1.0'

import logging
import sys

# python 3 doesn't have a unicode type
try:
    unicode
except:
    unicode = str

def set_debug(debug=True, filename='dicttoxml.log'):
    if debug:
        print('Debug mode is on. Events are logged at: %s' % (filename))
        logging.basicConfig(filename=filename, level=logging.INFO)
    else:
        logging.basicConfig(level=logging.WARNING)
        print('Debug mode is off.')

def xml_escape(s):
    if type(s) in (str, unicode):
        s = s.replace('&',  '&amp;')
        s = s.replace('"',  '&quot;')
        s = s.replace('\'', '&apos;')
        s = s.replace('<',  '&lt;')
        s = s.replace('>',  '&gt;')
    return s

def convert(obj):
    """Routes the elements of an object to the right function to convert them based on their data type"""
    logging.info('Inside convert(). obj type is: %s' % (type(obj).__name__))
    if type(obj) in (int, float, str, unicode):
        return convert_kv('item', obj)
    if hasattr(obj, 'isoformat'):
        return convert_kv('item', obj.isoformat())
    if type(obj) == bool:
        return convert_bool('item', obj)
    if type(obj) == dict:
        return convert_dict(obj)
    if type(obj) in (list, set, tuple):
        return convert_list(obj)
    raise TypeError('Unsupported data type: %s (%s)' % (obj, type(obj).__name__))
    

def convert_dict(obj):
    """Converts a dict into an XML string."""
    logging.info('Inside convert_dict(): obj type is: %s' % (type(obj).__name__))
    output = []
    addline = output.append
    for k, v in obj.items():
        logging.info('Looping inside convert_dict(): k=%s, type(v)=%s' % (k, type(v).__name__))
        try:
            if k.isdigit():
                k = 'n%s' % (k)
        except:
            if type(k) in (int, float):
                k = 'n%s' % (k)
        if type(v) in (int, float, str, unicode):
            addline(convert_kv(k, v))
        elif hasattr(v, 'isoformat'): # datetime
            addline(convert_kv(k, v.isoformat()))
        elif type(v) == bool:
            addline(convert_bool(k, v))
        elif type(v) == dict:
            addline('<%s>%s</%s>' % (k, convert_dict(v), k))
        elif type(v) in (list, set, tuple):
            addline('<%s>%s</%s>' % (k, convert_list(v), k))
        elif v is None:
            addline('<%s></%s>' % (k, k))
        else:
            raise TypeError('Unsupported data type: %s (%s)' % (obj, type(obj).__name__))
    return ''.join(output)

def convert_list(items):
    """Converts a list into an XML string."""
    logging.info('Inside convert_list()')
    output = []
    addline = output.append
    for item in items:
        logging.info('Looping inside convert_list(): item=%s, type=%s' % (item, type(item).__name__))
        if type(item) in (int, float, str, unicode):
            addline(convert_kv('item', item))
        elif hasattr(item, 'isoformat'): # datetime
            addline(convert_kv('item', v.isoformat()))
        elif type(item) == bool:
            addline(convert_bool('item', item))
        elif type(item) == dict:
            addline('<item>%s</item>' % (convert_dict(item)))
        elif type(item) in (list, set, tuple):
            addline('<item>%s</item>' % (convert_list(item)))
        else:
            raise TypeError('Unsupported data type: %s (%s)' % (obj, type(obj).__name__))
    return ''.join(output)

def convert_kv(k, v):
    """Converts an int, float or string into an XML element"""
    logging.info('Inside convert_kv(): k=%s, type(v) is: %s' % (k, type(v).__name__))
    return '<%s type="%s">%s</%s>' % (k, type(v).__name__ if type(v).__name__ != 'unicode' else 'str', xml_escape(v), k)

def convert_bool(k, v):
    """Converts a boolean into an XML element"""
    logging.info('Inside convert_bool(): k=%s, type(v) is: %s' % (k, type(v).__name__))
    return '<%s type="bool">%s</%s>' % (k, unicode(v).lower(), k)

def dicttoxml(obj, root=True):
    """Converts a python object into XML"""
    logging.info('Inside dict2xml(): type(obj) is: %s' % (type(obj).__name__))
    output = []
    addline = output.append
    if root == True:
        addline('<?xml version="1.0" encoding="UTF-8" ?>')
        addline('<root>%s</root>' % (convert(obj)))
    else:
        addline(convert(obj))
    return ''.join(output)
