/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
Taylor series for Y1 in extremums and zeros.

Generated by SageMath:

```python
mp.prec = 115
terms = 28
print(f"pub(crate) static Y1_COEFFS: [[(u64, u64); {terms}]; {len(y1_zeros)}] = [")

def get_constant_term(poly, y):
    for term in poly.operands():
        if term.is_constant():
            return term

def print_taylor_coeffs(poly, n):
    print("[")
    for i in range(0, n):
        coeff = poly[i]
        print_double_double("", RealField(115)(coeff))
        # print(f"{double_to_hex(coeff)},")
    print("],")

prev_zero = 0

for i in range(0, len(y1_zeros)):
    k_range = y1_zeros[i]
    range_diff = k_range - prev_zero

    x0 = mp.mpf(k_range)
    from mpmath import mp, bessely, taylor
    poly = taylor(lambda val: bessely(1, val), x0, terms)
    # print(poly)
    print_taylor_coeffs(poly, terms)
    prev_zero = y1_zeros[i]

print("];")
```
 */
pub(crate) static Y1_COEFFS: [[(u64, u64); 28]; 32] = [
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc749367c4c05aaa, 0x3fe0aa48442f014b),
        (0xbc58e5288bbf5fe3, 0xbfbe56f82217b8f1),
        (0x3c45adcf3e0230ee, 0xbfa0d2af4e932386),
        (0xbc1aef581a20d1b6, 0xbf73a6dec37290aa),
        (0x3bea0fd3a3179817, 0x3f7e671c7d1196fd),
        (0x3bdbfdc1a54825cd, 0xbf65429dc5a45612),
        (0x3bf2b56b5db9ac40, 0x3f517ab4afac0735),
        (0x3bebebd87be68c3c, 0xbf40b2d877d5a29c),
        (0xbbcf3ef5ec675b9a, 0x3f2eea7bbb907646),
        (0xbbbc5481911cf1b3, 0xbf1c3fae3b0ad706),
        (0xbb97d87196deccbb, 0x3f09d174c9ccecd9),
        (0x3b9ce9bed2434862, 0xbef7956ad3fe9783),
        (0xbb81a3285733db7f, 0x3ee5865d11ebb14d),
        (0x3b7c0f8e48b356f0, 0xbed3a27f1fd4ac6e),
        (0xbb601bb903ea5d53, 0x3ec1e742c1cf6acb),
        (0xbb208185a8aebef4, 0xbeb051d134cb0a72),
        (0x3b2d22383a2945e9, 0x3e9dbea846629267),
        (0xbb2d0bd2365f955b, 0xbe8b1a1682f5eb1f),
        (0xbb17f54155dbfd8e, 0x3e78b0b08d8ccddf),
        (0x3b0c6ae86d8ad6fe, 0xbe667d8a7cc18981),
        (0xbaea87f82b9f7fa9, 0x3e547c0e174bc402),
        (0x3ae6655ff087a6b2, 0xbe42a7fb50d94c09),
        (0xba74f97124d12cda, 0x3e30fd661ade45f0),
        (0x3aad4c923a16d045, 0xbe1ef1655b283bca),
        (0x3aa6328133433e29, 0x3e0c2d21fe733c5f),
        (0xba292d7e82d617f2, 0xbdf9a7efb711afac),
        (0xba60d9e2266f470a, 0x3de75c3fc5194fe2),
    ],
    [
        (0x3c3c696f3eb0ae77, 0x3fdaabb4011ed330),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc682b7fe4dd90ef, 0xbfc8b45babe797ba),
        (0x3bf255e280edf040, 0x3f8e147099a6f0ea),
        (0xbbfea368c6887e7f, 0x3f88c5af1eeb2d6b),
        (0xbbca04ffe9904e56, 0xbf4133fa47da52e2),
        (0xbbd3086cde21e2cb, 0xbf3bf8af93ff0b8c),
        (0xbbafa9018f253e3c, 0x3f021d64be62ccf9),
        (0x3b3a4d52bc5a61e1, 0x3eb44d2ce67b2d18),
        (0xbb48016de8c47acb, 0x3eb14c3ab2ad79cc),
        (0x3b28724eeca6e146, 0xbe9b8eee52e1ce01),
        (0xbb184699c450d4ac, 0x3e7a85f1a878746a),
        (0xbaf5fb00f2f338b7, 0xbe5bf86b3c89849b),
        (0x3adf14ae65986334, 0x3e3f01920a1ecb6b),
        (0xbac5a8492238e17f, 0xbe20fac544c5a674),
        (0x3a972f86dd69da9b, 0x3e02818841d3a289),
        (0xba686f565d0da109, 0xbde42b5ceddd3872),
        (0x3a540c6129af60a6, 0x3dc5f9472a38633d),
        (0xba4276328d0b688c, 0xbda7ed484e6442ee),
        (0xba23ab0f9748db4a, 0x3d8a0b10351f65a6),
        (0x39d5b91f2ce320af, 0xbd6c567ba5000a2c),
        (0x39e0b587d390388d, 0x3d4ed39e06bafa8a),
        (0xb9ca4dd40bbdfa57, 0xbd30c36f7f68d190),
        (0xb9ae9f2d1d6c950e, 0x3d123a81488a8c60),
        (0x39796d1b53f4c705, 0xbcf3d198a043957c),
        (0xb97e66e460933260, 0x3cd58b80d1da3cb2),
        (0xb95de7856c390662, 0xbcb76b414bca83a9),
        (0x3923face03bca6a1, 0x3c99742311b12a8a),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c7b8d2a1c496808, 0xbfd5c7c556f0c19a),
        (0xbc370c18fffb661e, 0x3fa00b9f8571ca1f),
        (0xbc4e4289c3acba27, 0x3faa15d92dfe3e27),
        (0xbbf5a3591cb63f33, 0xbf710a329e2c23fa),
        (0xbc0511883ce39919, 0xbf61be6db9923acf),
        (0x3bcb8759d4a0e5cd, 0x3f2337c7e138f484),
        (0x3ba7299907aec5c4, 0x3f085b940eb607f9),
        (0xbb2063eb830ecaea, 0xbec80619147b78f2),
        (0xbb4468879e6c349e, 0xbea255e6cf70cf33),
        (0xbafc601de5ef1462, 0x3e5b62cd02014989),
        (0xbad6adc77b77235d, 0x3e380b1aac007d1d),
        (0x3a624f45defeeef1, 0xbdfa7ee05a568fe6),
        (0xba283df15edc8775, 0x3d84065ca5f23bb5),
        (0xba2c1a2ebca15779, 0xbd93847684c7da93),
        (0x3a08225b40e694f7, 0x3d74980895b5067a),
        (0xb9eed81f16f01fb8, 0xbd4c00f7d6be89d7),
        (0xb9a4e2f268d1912d, 0x3d2402fd58efd6fd),
        (0x399973b6338a6d0f, 0xbcfded82b76f72f2),
        (0x3978023c06cffc92, 0x3cd6380d6ba31980),
        (0xb94735a17d43c8d2, 0xbcb06b38ec70e0a6),
        (0x3907f71a9df6b85b, 0x3c88443f14394c1d),
        (0xb8f01075b123018b, 0xbc61ee2105252260),
        (0xb8b57669955b81a8, 0x3c3a7c72c37961bf),
        (0xb8954dc89e6e0348, 0xbc138e5e595f9137),
        (0xb87ee16a4e8bff5b, 0x3becdf0d22c4f792),
        (0xb86abb1732ed43be, 0xbbc54e9dcbced4f1),
        (0xb82e4e62e1a676d6, 0x3b9f718438e6ba2f),
    ],
    [
        (0xbc583acf6511df31, 0xbfd36732d4b96094),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc5b435291c33433, 0x3fc3001c8002caf8),
        (0x3c085153ba2c907f, 0xbf7bf5a03bab4999),
        (0x3c28d66c540219c4, 0xbf8751ea028c1953),
        (0x3be6d2f429037cfc, 0x3f423874cd8d0401),
        (0xbbb4d7b925522373, 0x3f364f6610d64939),
        (0xbb9920eb782e7608, 0xbef02978de3838b9),
        (0x3b66c8e5402ae5c0, 0xbed72f0766d0d591),
        (0x3b20b724f28fabb4, 0x3e8f2081874c7e33),
        (0xbafe89c3eab7b761, 0x3e6defd5dce00666),
        (0x3ab712e9adc751d5, 0xbe2205c6fe8f26fd),
        (0xba90a1fcff20092c, 0xbdfb6432e005a435),
        (0xba40d93a2f6bee54, 0x3db0289fce476883),
        (0xba2c7d1344d3ff23, 0x3d80778a2d37fe43),
        (0x39a7b4208bcbde0d, 0xbd29723fe5684800),
        (0xb9a84df62153f4f5, 0xbd082f3f70663f33),
        (0xb91998f4a32b0e4b, 0x3cc7a09f40ec7306),
        (0x39256931314bf68c, 0xbc85319ec342127c),
        (0x38fd4eda573e2499, 0x3c62d63483483be2),
        (0xb8a4a42a37750699, 0xbc38c3ecf9381019),
        (0x3893811d88a13751, 0x3c0ba15f09ca6382),
        (0xb8653a606cd73ccb, 0xbbdf92174ea83693),
        (0x385bcb6c1b8ec617, 0x3bb25604c3a40365),
        (0xb82168dd0f69d5f1, 0xbb853b55eb1a54fe),
        (0xb7e96fc33c8ede98, 0x3b5889d72e8200c6),
        (0xb7b3762a6112ab36, 0xbb2c5b6e1cc194ec),
        (0xb78614198a535866, 0x3b0061d24b95ed7a),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c61dc672a53c590, 0x3fd15f993fceab5c),
        (0xbc1455aa7db14f38, 0xbf902b3933cf21b1),
        (0x3c443ede64b782d9, 0xbfa6395dfe49fcd4),
        (0x3be10ad2b71b9820, 0x3f63ced2a2e69180),
        (0x3c0f30cad7e0ebe8, 0x3f607a678d6000bb),
        (0x3baf8fd02659f1ef, 0xbf1b50d7e1d32020),
        (0x3b91fb17827843f5, 0xbf06f7bab104f355),
        (0xbb60ab92724984c7, 0x3ec176e72bf94b8f),
        (0x3b31557f6e92a4a7, 0x3ea2becb2b6bbc6d),
        (0x3aea30e9dd0c4fab, 0xbe5a384eebfe1367),
        (0xbad27d53b702d9ee, 0xbe341e7a923c3a90),
        (0x3a8be8710ba9cded, 0x3de9e3284f5dd101),
        (0xba5a7ff332888842, 0x3dbec40b623ea350),
        (0x3a1bb0f2b7e06e32, 0xbd726868b83f8d7d),
        (0xb9d127448ae5dbb8, 0xbd416f7ae80cad77),
        (0xb98b4267f56d6b60, 0x3cf318658ecb9bd5),
        (0x394a28f7ccb2ac45, 0x3cbfacabdbabe483),
        (0xb8fdeb4ca08fc098, 0xbc713cc1f6930364),
        (0x38c8e1a7a7c446ff, 0xbc33cb94044d8aee),
        (0x3846a7a1323190de, 0x3bda22029099f534),
        (0x385ab5c5889916dc, 0x3bb47d9b88989e97),
        (0x381dfc385d309d52, 0xbb745299177606ed),
        (0x37def84f6911654b, 0x3b3503d9701ead82),
        (0x3757dbc183f5d86f, 0xbb08176b7514ca80),
        (0x3726e80a82be2e11, 0x3ad863cb123482cb),
        (0xb7400c9d5c314e72, 0xbaa6552586051784),
        (0xb6ed950d76fdf6bc, 0x3a74afbafce97c21),
    ],
    [
        (0xbc64f19a2762f5ae, 0x3fd00ef3745e0e3c),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc2a5ca3437ac1c3, 0xbfbfcdacdda138f2),
        (0x3c19a8e604f93931, 0x3f706cc34cd829fa),
        (0x3c01d3813b5701a3, 0x3f84641bb10c16cb),
        (0x3bbd08ac252b4932, 0xbf37fac943e2a16d),
        (0xbbd98c55ed55e5ec, 0xbf34769ed32e14a2),
        (0xbb8b665d8b88b73b, 0x3ee80608ecda1520),
        (0xbb7d207258864fea, 0x3ed5cc8242d77e79),
        (0x3b2f5c5f133a954a, 0xbe888c8f253923cc),
        (0xbb0d50fa8d8c7a72, 0xbe6ce5908c1f1376),
        (0x3abc19806933366e, 0x3e1ed1625825eeeb),
        (0x3a969b1553c1a884, 0x3dfa30d624f10edf),
        (0x3a41711a2b894301, 0xbdaa50765ef82dbe),
        (0x39ef8b9861fc6da9, 0xbd814cd364ef8287),
        (0xb9bec99ce5124af1, 0x3d306199db2399a2),
        (0x39a24cb0ca5a627f, 0x3d0164150f087360),
        (0xb90883d74ed9e2a1, 0xbcaefb1085d3b5a6),
        (0x390110bd90901ca7, 0xbc7ba1419f5a6f40),
        (0x389f7ac7f08169b1, 0x3c27638ccf29c1b0),
        (0xb89f51222b4b42bb, 0x3bf18a62dccdca9d),
        (0xb83b3be96059d0b4, 0xbb9b54a4a2d7343a),
        (0x38017f10a44ce3ed, 0xbb635931e0eacdee),
        (0x37be3118e01854ca, 0x3b1036817e50e3af),
        (0xb7657afa36f3f4a9, 0x3aca87e3e3debd9e),
        (0x36c384606510e7f7, 0xba44ec6558f8bb31),
        (0x36c308ed950cf583, 0xba4d684c0820785d),
        (0xb6acfa02e081161c, 0x3a0f0d58bca1c41b),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c54d14c77bc1691, 0xbfcdc14ea14e89f9),
        (0x3c2340bd04e7df4a, 0x3f84429fef5b5fbd),
        (0x3c4f19a8f5b1379e, 0x3fa367d7d608e4ba),
        (0xbbf4eb6bddef363a, 0xbf59d6eb2bc49e35),
        (0xbba2f121221c388c, 0xbf5dc4f991b3db86),
        (0x3bbf047b90425584, 0x3f1315ec04d6e6bc),
        (0x3b9087e84c57d2ad, 0x3f0571814a1aa301),
        (0x3b2795eb85278f1d, 0xbeba2977fa42f227),
        (0x3b3f92ddd9da4ad7, 0xbea1e86423086328),
        (0x3aff19e1c5d9e237, 0x3e54a7b82d41f690),
        (0xbad0197985af09ca, 0x3e3390660a1bb74c),
        (0x3a8a68e5322dab0d, 0xbde549e8b3ed12f4),
        (0x3a500e54fb5a85ec, 0xbdbe32cf77b347a3),
        (0xb9d3ea446c95f85a, 0x3d6eff58d55a39fb),
        (0x39d1c41d97727556, 0x3d415e5c0af749d4),
        (0x3994a6959d259d8b, 0xbcf0d67f939f4917),
        (0x3951f0d6f37c7a4e, 0xbcbef3f2afde9972),
        (0xb8ce371e6c4a5fb7, 0x3c6c60ed345061d2),
        (0xb8ce6598cd4716cf, 0x3c360519c76d6d8a),
        (0xb868ddc08c092699, 0xbbe324831e767061),
        (0x3820a7a0186edebf, 0xbba99201cd6f64ed),
        (0xb7d50b99e3cbd8ed, 0x3b55120660d9db07),
        (0x37bd05b3e6de1af6, 0x3b18d03fd947900e),
        (0x373b419f47dccb93, 0xbac397b2f45393ab),
        (0x371a75815e071b8c, 0xba840af84bf4566b),
        (0x36c253d300be5f73, 0x3a2d23b6405995b9),
        (0x365d14598eec4332, 0x39ee07cb3c4c18f5),
    ],
    [
        (0xbc5a8046704071bd, 0xbfcc075da85beb4f),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc53b0c30f3f6eaa, 0x3fbbdeb6ff9f55e1),
        (0x3bef54869c1812be, 0xbf661eefb74da882),
        (0xbc20492140a086db, 0xbf8229ea914b846e),
        (0xbbc9a664ae9a2c1e, 0x3f30cbcc6778fd37),
        (0xbbd8078d46437190, 0x3f32aa59f5091f7b),
        (0xbb7b81f3aff07289, 0xbee1c15d5251ae6a),
        (0x3b707e9f1d7b7e6a, 0xbed4583f15abd692),
        (0xbb2cc9b93ff3f6ce, 0x3e831d151a1284aa),
        (0x3b0aa58bdebda7d1, 0x3e6b74e57c226d3c),
        (0x3abe0f534111cac9, 0xbe19044f1378d3e4),
        (0xba92c7e898648f68, 0xbdf93b1ec7cafe33),
        (0x3a314e62ad0639ef, 0x3da61a4e89bf8438),
        (0x3a298cd71751f9aa, 0x3d80d430f7e78fff),
        (0xb9c643be1c5347dd, 0xbd2c3b0a8f7a7b0e),
        (0xb9608e40ab9aafb9, 0xbd010e773677ab43),
        (0x3946ed856edf6b25, 0x3cab59410f7e6f10),
        (0xb8e86e49965ef8df, 0x3c7b2e693f06a801),
        (0xb8bd6df9232d394a, 0xbc24d1b6b5dbdd8a),
        (0x389d45bfc9e28c95, 0xbbf17c6418e80b27),
        (0x38330cb81415d134, 0x3b999a3de4a881f9),
        (0x38014861b029a2c1, 0x3b628ca7ea60d3ed),
        (0x3763e7968eedb33e, 0xbb09f8cd292fbd53),
        (0xb772e1ee82abca63, 0xbad0840446d9db33),
        (0x36eac14fda382bcb, 0x3a7627f31e8af8f1),
        (0x36d03dd271f85894, 0x3a38fffe9d4c8a9c),
        (0x368992b4a1b37e32, 0xb9e006ef0777e888),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c57ba12cd0fc91f, 0x3fca7022be084d99),
        (0xbc17334559c5138c, 0xbf7c650b6b83109a),
        (0xbc4d0f8f36713120, 0xbfa163191c30aa62),
        (0x3bbcf7c95031f425, 0x3f526b045287ddca),
        (0xbbf7b71420f279ea, 0x3f5b17602840abf5),
        (0xbb99a95b35521d32, 0xbf0c0a9cee3c842a),
        (0xbb91505ce3b29a78, 0xbf03e398cbc472ea),
        (0x3b5125ae0677178c, 0x3eb3f35db1ff1b8e),
        (0xbb4e582cdfc5009b, 0x3ea0e9b612dbd385),
        (0x3ae57a34bc6cd52d, 0xbe5056babcd9632b),
        (0x3ad18fdd63164b8f, 0xbe32c1a8c8f963a5),
        (0xb9e37fef15ca364c, 0x3de161b6aa9fde63),
        (0xba4ed1a8c48bfaf3, 0x3dbd4caa1e0162bc),
        (0xba0907496979f354, 0xbd69fdda132e53b1),
        (0x39dc92d876d25063, 0xbd4101c31e1df223),
        (0xb980f5f389d54392, 0x3cecdc25a81e55f3),
        (0xb9476a7e07116985, 0x3cbe87025cca1380),
        (0x38f93a72b93872b3, 0xbc68c602211c7544),
        (0xb8dd598494fb4eb0, 0xbc35d4a3a29a7da9),
        (0x387e1025298516ea, 0x3be0f3be0d4b8ee0),
        (0x3829d1ee76262ed4, 0x3ba97a197d3cf12a),
        (0xb7f42ff2642fd193, 0xbb52f220fc5e4bf1),
        (0xb79a001d511e0971, 0xbb18bb200c3f6737),
        (0xb75720aadc56d8cc, 0x3ac1a0fc9e585d08),
        (0xb6f810019c65facf, 0x3a8449eb5d1ccd0b),
        (0x36c0524fdb6e5bf4, 0xba2bc209d14d289f),
        (0xb663781b8b017ac7, 0xb9ec816d53cd34ad),
    ],
    [
        (0x3c518b0303bbe2f5, 0x3fc931a5a0ae5aa0),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c348c3f5f957887, 0xbfb919c8a3f203fa),
        (0xbc0ef6b166c80aea, 0x3f602a38da6262a9),
        (0xbc207165718dd6e6, 0x3f807ced48910819),
        (0x3bc68f5ae1c2651e, 0xbf2900f33a00690a),
        (0xbbc8c05ccff0a4b4, 0xbf31278d46fd153c),
        (0xbb6f62aece58de7b, 0x3edb2595529cf1c7),
        (0xbb731bd5840756d7, 0x3ed2f7c2d608e0eb),
        (0xbb1f324b35355b3a, 0xbe7e212d2378b576),
        (0xbb005c49ab958b73, 0xbe69f3fcf3638b5a),
        (0xbab2a5d22fb87422, 0x3e144fbf0377cdfa),
        (0xba636fc80c961f59, 0x3df82268e83f4ed0),
        (0x3a40f5458ba7db6a, 0xbda26cc2b02dc737),
        (0x3a2ce24e6964787f, 0xbd80418bae8bbc48),
        (0x39b27ec279e7b798, 0x3d2812815e643e0a),
        (0x39a36cf3023b50ff, 0x3d009b2ac849ff66),
        (0x39133d4af08ad917, 0xbca7c55d0a6d0728),
        (0x38de47b036357356, 0xbc7aa167b939df6b),
        (0x3886507c635a91ef, 0x3c22645687164ddf),
        (0xb88b93b25f237a68, 0x3bf137648ca710a0),
        (0xb8159c0acc3fb928, 0xbb96ed64da2728ee),
        (0x37e7e2b02a3ea356, 0xbb62564f89aafafd),
        (0xb7a8f4b2647af4ab, 0x3b0789c5abd21586),
        (0xb7722d480745c160, 0x3ad05f7f4ebd3555),
        (0x37185f3d16f70670, 0xba7442d6813292c4),
        (0xb6d2d55a451fd98e, 0xba38df199441f62a),
        (0xb65428bef2d53556, 0x39ddaedad9d103ec),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc63db68c567283b, 0xbfc80781c32422e7),
        (0xbc1bf83906fd50c1, 0x3f754eda697a0098),
        (0x3c232419a9d405db, 0x3f9fbe6df840847f),
        (0x3bd5a76a98bd9674, 0xbf4be318d61276e1),
        (0xbbe633162d1131fe, 0xbf58efee4094379c),
        (0xbb4139e1a56d06b2, 0x3f059145b4f0e4de),
        (0x3b2db153b13f2cee, 0x3f0282d26a74c38e),
        (0x3b48c60f3e751263, 0xbeaf56c29d9ad959),
        (0xbb2ffef5e5f8ef91, 0xbe9fdd03174f902a),
        (0x3acf87aa0c95189d, 0x3e4a44a7907fee59),
        (0xbaded732eaed6428, 0x3e31df65332ab3e4),
        (0xba6ed0d6f75e03b2, 0xbddc96e9cf361a43),
        (0x3a2be09d928f0cb2, 0xbdbc3439f3fac67e),
        (0x3a0894702a4e04ba, 0x3d65d38ae50afdda),
        (0x39c847e4ec1195d0, 0x3d40833ea1f83461),
        (0xb9816bc0dd8ea673, 0xbce8afb70fb3c9e8),
        (0x395196a0917179fd, 0xbcbdda411e59ed8f),
        (0xb8c4610e8c50f3dd, 0x3c65889665db09a8),
        (0xb8d125a6e60383eb, 0x3c35792846a5c5e9),
        (0x38475c9e6c26184f, 0xbbdde1c2aac4bc04),
        (0xb834983a005c0922, 0xbba92deda741b574),
        (0xb7fd331deb45d073, 0x3b50e62611579a41),
        (0xb7b4d43c6fbeedae, 0x3b18898163cacd25),
        (0x37345581502d5210, 0xbabfc6f37553b483),
        (0xb72eac4aa67583bb, 0xba843147e449ee48),
        (0x36a768c097b2d633, 0x3a293dd77e784298),
        (0xb667f8f81d88a65c, 0x39ec7209a4872362),
    ],
    [
        (0x3c559364e26d93ec, 0xbfc713fc51664c74),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc488077fd102b67, 0x3fb7049760cde490),
        (0xbbff945ad3318fed, 0xbf58ef5f1cbe4874),
        (0x3c0b034f6fcde568, 0xbf7e5f53caf3bead),
        (0x3bcb16596420523c, 0x3f237b0b62ddadd1),
        (0xbb7411414067003c, 0x3f2fd3bac08286da),
        (0xbb66bf25df78108f, 0xbed5789803de3afe),
        (0xbb30f115b8c73e8d, 0xbed1c0faa89993b9),
        (0xbb1173613a4d69f6, 0x3e7845b49b0674ba),
        (0x3ada42fc2e50738e, 0x3e6886872801396c),
        (0x3a9cf261fcd9b5e3, 0xbe10b036779ac071),
        (0xba8ed0f2c9abc118, 0xbdf7049d18354c88),
        (0x3a3b53ce13beae6c, 0x3d9edd9d0fb2ebd2),
        (0x3a01c67e454c422c, 0x3d7f44600957fa58),
        (0xb9c8cf2dd5194971, 0xbd2486b1d6a91df8),
        (0x39a50688f914608c, 0xbd0016025abd20b4),
        (0x392da82dcfc4875f, 0x3ca498b3d1754fd7),
        (0xb9082f62c6ce39c5, 0x3c79f47baf518ec1),
        (0x38a6105821b4975c, 0xbc2029b685b728de),
        (0x389216a765538354, 0xbbf0ddbdab448097),
        (0x3830ae5354e24a65, 0x3b94667d2d37a045),
        (0x38016f3b498e4738, 0x3b620a8a21b80dcb),
        (0xb78daf501176b370, 0xbb052c2caec48eea),
        (0x377fe5ca7bc39984, 0xbad02a81f53376b9),
        (0xb714e540400d2a70, 0x3a7266275dfcd434),
        (0xb6a1501fbef1dde4, 0x3a38a1141391318a),
        (0x3672180bf6683e02, 0xb9db2e3c29487cd8),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc6d2f0105f3ce7c, 0x3fc62d94d97e859c),
        (0x3bd10ece6e29b084, 0xbf70bf614807033c),
        (0xbc37013075a066f9, 0xbf9d5f857a2a6107),
        (0x3be49380eb03f78d, 0x3f46081b0b7fe572),
        (0x3bfce3676d17b40c, 0x3f57307b03e248f8),
        (0xbbabc5fb92b99e67, 0xbf0132c0aa83d0dc),
        (0x3b962d899f08a704, 0xbf0154ed4598d2f0),
        (0x3b436681f34b2420, 0x3ea94f64f476e615),
        (0xbb2a69d5b51b6d77, 0x3e9e12725853a9fb),
        (0x3aec161d9909f167, 0xbe4588c758dfc8ab),
        (0xbade3ba8f2d73fb0, 0xbe31021cdd9b5f7f),
        (0x3a7b80d7aefb24c5, 0x3dd7cfa7a6c26b53),
        (0xba5b73dd1f1185ae, 0x3dbb0e011d23f5e3),
        (0x3a0a3aee2b581931, 0xbd6276cc0fa65a6a),
        (0xb9d2974c386cdc2a, 0xbd3fe92f0882b440),
        (0xb982bfc89a29ddc4, 0x3ce53126d86bf487),
        (0xb94dd068a7b1e2e7, 0x3cbd07f6c1993c8c),
        (0xb8fb38430712c4a6, 0xbc62bc249a86f7c9),
        (0xb8d065bd6a2ed972, 0xbc34ffb4f0789e1c),
        (0xb854136ccbc577a4, 0x3bda504d6417aae1),
        (0xb84be408189a39ee, 0x3ba8bd8d3bf76896),
        (0x37eab1c3ced9a78a, 0xbb4e157fb142efc6),
        (0x37a40d8e8cc67990, 0xbb1834ef259f93cb),
        (0xb75167300d24b2f1, 0x3abc8e7b497b27c6),
        (0x36d98f27f9bf4556, 0x3a83fd162f3a5584),
        (0x36c5a4266d838510, 0xba26e0585c2f983a),
        (0x366678ba75bc0bf4, 0xb9ec3d3ba73f49c5),
    ],
    [
        (0xbc5b858aeca267e1, 0x3fc56b97f8091ac5),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc21d2a41bfc4c1d, 0xbfb560fcc8c08469),
        (0xbbe6529450b1ff0b, 0x3f53fafa39618883),
        (0xbc1ad54776e42e6f, 0x3f7c49141623372f),
        (0x3bb39ecd4785a672, 0xbf1f69980694fd17),
        (0x3bcc2a33b89df073, 0xbf2dc5f848aa9d34),
        (0xbb7ed412867be695, 0x3ed178fc979b77bd),
        (0x3b73a93300f0b087, 0x3ed0b494a4bafcc7),
        (0xbb1a1471efd3f084, 0xbe73fc3884c2743a),
        (0x3b0b5eabaa159e81, 0xbe673afb9fb4d844),
        (0xbaa47c72ef2556f3, 0x3e0bd90346a3e0ed),
        (0xba99f08140c074ed, 0x3df5f3bafb215b44),
        (0x3a2ab2b535dc2fdf, 0xbd9a1c27c37569c8),
        (0xba09f88e2bb9d462, 0xbd7e0455dbf9cc4f),
        (0xb9cdb0302c98b2dd, 0x3d2199b19fcacda0),
        (0x39851865cfbaca3e, 0x3cff1334609dc3bd),
        (0xb8c2cd00a5fcb440, 0xbca1e37265d5ce04),
        (0x38c8d0340828b4e4, 0xbc79353f1aee0a50),
        (0xb897b28613de5d51, 0x3c1c68e4abc12011),
        (0x388b3ab1625e378c, 0x3bf075c553243bdb),
        (0xb82923cf876dbeab, 0xbb921f6f50442923),
        (0xb80ef47dee3f5afd, 0xbb61ae48340c53aa),
        (0x37a8614c654ed794, 0x3b02fdf33e2b9584),
        (0x376126c601c703f9, 0x3acfcd13a6548328),
        (0x371d5986e5068b9e, 0xba70a5f4affbe604),
        (0x36c041d0dc32e320, 0xba384ccca4432680),
        (0xb652e0a78e43b84d, 0x39d8c95636d09811),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc5e9088e9ff2518, 0xbfc4b2a38f1ab9b4),
        (0x3bf658189763e4b9, 0x3f6b3878aadeb34d),
        (0xbc3525fe797e2126, 0x3f9b750d89a9b35f),
        (0x3bb2c48be700f9e6, 0xbf41f6911725a956),
        (0xbbed0a355d52c196, 0xbf55beee6fd51c8a),
        (0xbb8079c92f6ce1f4, 0x3efc3625d7a65089),
        (0xbba1cffb5ca00bc2, 0x3f005375a588a72b),
        (0xbb398a011960416e, 0xbea4ee5e4e7cb18b),
        (0xbb2804ad812748e6, 0xbe9c7b3d81b5ff94),
        (0x3aea91d5277c5345, 0x3e41fce14f48518c),
        (0xbadb70b3ae1d3fb1, 0x3e30346643c9d86e),
        (0xba7397dad107b33b, 0xbdd41c861bc1c34b),
        (0xba5410a8098eafad, 0xbdb9eeda214eabbe),
        (0x39f8372842605aa2, 0x3d5f8cf277ce6338),
        (0xb9de510926654c46, 0x3d3ec46e14cd5d9e),
        (0xb962703fb3cb397d, 0xbce2500181b463c7),
        (0xb95b2c04b73484a3, 0xbcbc24c2b350e45a),
        (0x3888f6781f273b56, 0x3c605d9c149f1b8a),
        (0x38d31e5cc968c925, 0x3c34752e1249314b),
        (0x387b7657eab8f7e8, 0xbbd73842b12b31ea),
        (0xb846e57a01049463, 0xbba83594fd823055),
        (0xb7e962546fd68b46, 0x3b4acb92f062a2fd),
        (0x37beb28a448a691b, 0x3b17c779242729e1),
        (0xb7536df5c047d7ba, 0xbab9a6de3d6e1877),
        (0x36e0ce403e109088, 0xba83b3c68747091c),
        (0x36bd2b9ab8c7eb26, 0x3a24b55c31763430),
        (0x36831011a1450327, 0x39ebeab3c6cb6a58),
    ],
    [
        (0xbc4d00ae4313eed0, 0xbfc413644356a52b),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c2e7840863e7c03, 0x3fb40bb88c6f2b85),
        (0x3bd5631193c7aba5, 0xbf5078d13cfc400e),
        (0x3c1a406bffcdbe35, 0xbf7a9191262ab9d5),
        (0x3bbe2809cb431b0a, 0x3f1a005297618f35),
        (0x3bc92fd90e2b1396, 0x3f2c0cbad847a60e),
        (0x3b6faeaea4946f0f, 0xbecd1a72e7c35fd8),
        (0x3b6a2e571e6832b0, 0xbecf9a2654099c40),
        (0x3b14a1a09cd63457, 0x3e70c6b06e20fcbe),
        (0xbb08c15da7295060, 0x3e66136d6425ea74),
        (0xbaa9c15db46970d1, 0xbe07977677c6badc),
        (0x3a91f266c960af6f, 0xbdf4f77b3143a31f),
        (0xba2ade992d25ffbb, 0x3d965720b0a36de3),
        (0x3a0635bc57fc436d, 0x3d7cd126dc5db8d4),
        (0x39b0b42ceb8837b9, 0xbd1e6d575a54f608),
        (0x399c1abf000da8a7, 0xbcfdfb4d0beacd2f),
        (0x3926e0efc20cf902, 0x3c9f3cccbe3a6b08),
        (0xb90208f2daceba75, 0x3c786f5c7a82aa3d),
        (0xb890e01b243ac14b, 0xbc190b8ec99be279),
        (0x386588585f7010ee, 0xbbf0061c1563865c),
        (0x3839090c97be6541, 0x3b901f2cce387f59),
        (0xb808c405323747f4, 0x3b61476b2aa5f9cd),
        (0x37aa73fdd9180681, 0xbb010995cf568203),
        (0x376999ab5e964dc7, 0xbacf2f8eadcda35c),
        (0xb6eef0a26c086d84, 0x3a6e19ecc98ab1c0),
        (0x36cbb3d7e3e4912f, 0x3a37e713fd44413e),
        (0x367917d21157a7a9, 0xb9d6910a2ab8996f),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4997782859a00d, 0x3fc37aaceac987b9),
        (0xbc09de167b0f4e45, 0xbf66afe4fe0bc0f7),
        (0xbc3d91e80a0529b8, 0xbf99de7a33bc3a97),
        (0x3bdacaad4f266959, 0x3f3e024f567ac487),
        (0xbbf5aee36a6b58af, 0x3f548843c426abe0),
        (0xbb8ed7ac2e5eef51, 0xbef7a8e14711c0f5),
        (0x3b9b3675770103a2, 0xbefeeceb341ad833),
        (0xbb4d96e3726387e9, 0x3ea1a743e05b39ca),
        (0xbb36e8969001b255, 0x3e9b143d39c90dd5),
        (0xbad319f8c4237007, 0xbe3e8e00012321d3),
        (0x3aae89c8a788cf46, 0xbe2ef28e323e7496),
        (0x3a7f30cb64f4f0d2, 0x3dd137a1bf5bde2c),
        (0x3a5bf7dee3b649de, 0x3db8e087c7cb7f1d),
        (0x39fb7806a5d33650, 0xbd5b3dc9dd36d846),
        (0xb9c8eb799dd68c33, 0xbd3da6ae03deea14),
        (0x394156b16e9a68f1, 0x3cdfe64e83599736),
        (0xb93c20b358422ecc, 0x3cbb3e899a8275b8),
        (0x38e23edcbf53670c, 0xbc5cc15471c3a791),
        (0x38d6153c40fa8364, 0xbc33e34956740ffd),
        (0x387da1d09b7c1150, 0x3bd491cabcac2ce3),
        (0xb84c9d6b2a07c067, 0x3ba7a07e1fc0dd64),
        (0x37bbd51dd0af2201, 0xbb47ecb9d1547f7f),
        (0x37bc94c2359a7968, 0xbb174a2cfb1be55e),
        (0x375ec790d520dd3c, 0x3ab712c909b7eb19),
        (0x3721fa30708a2d3f, 0x3a835bca00a6162a),
        (0xb6b4073920a6f8c9, 0xba22c13871e75a28),
        (0x368fbab0e8ced2dd, 0xb9eb82223e5c0791),
    ],
    [
        (0x3c4f123683cf20a8, 0x3fc2f4e70d6c7e01),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc5702b97938a87e, 0xbfb2ef24d6f7526a),
        (0x3be17c9efd9928a1, 0x3f4bc33c9dc6ec82),
        (0xbc153a673a682d1a, 0x3f7920414ee2acbe),
        (0xbba8c8602e5501dd, 0xbf15f9173916a219),
        (0xbbbd01332ea23750, 0xbf2a94fdbdcec471),
        (0xbb6340ead67ee8d4, 0x3ec8b309990f950c),
        (0x3b62bfaf2a2bd5b2, 0x3ece087ff4517c02),
        (0x3b05bdd16bb39ef5, 0xbe6ca22ab12cb4b7),
        (0x3ad4981496e99ee8, 0xbe650d1f28635cda),
        (0x3aac3cc89e59b451, 0x3e04441552e259a7),
        (0xba95d08c5f0c93c3, 0x3df411b8a8258957),
        (0xb9eff3ddb842e46e, 0xbd9354e914c31b45),
        (0x3a11319e6211da5e, 0xbd7bb16f111ebae7),
        (0x39af5cff5d440351, 0x3d1a888536e27439),
        (0xb9967f728b3f0a85, 0x3cfced726c1026c9),
        (0x39328206ff6f6ec6, 0xbc9b745d75ff6377),
        (0xb91cb1d8bb8b1e84, 0xbc77aaea63220bec),
        (0xb8b1d26bdb3cb23f, 0x3c162ef6548b0ef1),
        (0xb85086d51f21de1e, 0x3bef27eac447efa0),
        (0xb81b0359e4922880, 0xbb8cc635a651ed03),
        (0xb7c265c1a580ae14, 0xbb60db1d9fbb3868),
        (0x379bfffc557e0311, 0x3afea050ba5a6c00),
        (0xb7616b89acbe6940, 0x3ace84aeda9b4b7b),
        (0xb690317b7234e3f2, 0xba6b3ca6e927e7ca),
        (0xb6d96744c0e38388, 0xba37754b584db1b4),
        (0xb67873c14331a4ce, 0x39d48c23ea283d56),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c6b7326e3fbaa70, 0xbfc2740819f1caaa),
        (0xbbad91d196951b1e, 0x3f6349369dc780bb),
        (0x3c211249d3675d98, 0x3f98868d7401bf2e),
        (0x3bab901426ad555e, 0xbf398cd1bebe1445),
        (0x3b8e845cf46e9c36, 0xbf537eef9aadeee2),
        (0xbb978bd7232789d5, 0x3ef43394c95b2d2b),
        (0x3b98d34686b724d9, 0x3efd6dfcdb026028),
        (0xbb39039620a1668b, 0xbe9e448fbc8a1f43),
        (0x3b1ab8335f5b2e44, 0xbe99d764ee07c839),
        (0xbad24a385ba857b8, 0x3e3a53958c8d74ff),
        (0x3aae692fcd7541f7, 0x3e2da0e1c8a08d37),
        (0xba682664a5ef0aa2, 0xbdcdd7f918de776f),
        (0xba55452efda66af5, 0xbdb7e68037d5c861),
        (0x39e92d13e5b8bdbb, 0x3d57c2e63bbe27ca),
        (0xb9b76fb98c811c8b, 0x3d3c973a175ffbb6),
        (0x3961655ae92c79f4, 0xbcdc033788921736),
        (0xb93459b0496214b2, 0xbcba5d9f2bdd8af3),
        (0xb8c9fbdcd265117f, 0x3c596cb946e78db0),
        (0x38ca3b3e172d732c, 0x3c335090afbe104f),
        (0xb87ce7df25d16bb8, 0xbbd24f9089431e12),
        (0x38470e95a1733e58, 0xbba7060105333e1d),
        (0x37e148c26a9d2e03, 0x3b457032c08fa72b),
        (0xb7bcc5adf699f555, 0x3b16c43d570a329d),
        (0xb7472f3277e653fc, 0xbab4ce5eeb2252e9),
        (0xb72e4e66be44e7f5, 0xba82fa99d3f60727),
        (0xb6c49cda263bbcc6, 0x3a21035a54fa4f71),
        (0xb682adf1bab34466, 0x39eb0a7a1202f74d),
    ],
    [
        (0x3c34f78a7cfd8a8a, 0xbfc20198200b699d),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c1ede75d6e0c0b0, 0x3fb1fd242a74e630),
        (0x3be31a446a0673c4, 0xbf47cf261dfbf19a),
        (0xbc0b11e38b0823f1, 0xbf77e4820ec1dde4),
        (0xbbbbba7e6db23817, 0x3f12e1bd281dfcbb),
        (0x3bc48aed48d0a248, 0x3f2950bb06c6fdf9),
        (0x3b5f50566d2c2881, 0xbec54a38ab6af546),
        (0xbb69fdc0d6c27c0b, 0xbecca94f38024fdc),
        (0xbaf76dde5144d48d, 0x3e68c7e75971c85c),
        (0xbb0678751b2ddc2c, 0x3e6423fc7e251ca9),
        (0x3aab4294ffe166f7, 0xbe019fe1d8e51b96),
        (0xba976b836ebd0291, 0xbdf34198c795142c),
        (0x3a1b6fa7e1df9192, 0x3d90e78cdab26dd9),
        (0xba1a2a34bc1f8645, 0x3d7aa74cb5ba4b0e),
        (0xb9a65f2dd9b4219b, 0xbd1756ee8369797d),
        (0xb98cae4f94465269, 0xbcfbee3ccbac62a4),
        (0xb93edac05af99c66, 0x3c984c93627c1364),
        (0x39192f873368f6f8, 0x3c76ecd2977e750f),
        (0x3893a3df310cde1b, 0xbc13c1ad2197e738),
        (0xb83e79b62fc45f90, 0xbbee45c9096ecdab),
        (0x3808f4906192c428, 0x3b89c8e8d1127213),
        (0xb7d635517d589b2c, 0x3b606d492670f0ba),
        (0xb772bd51822a155e, 0xbafb9c058bf87bc8),
        (0xb755af5326fb8ade, 0xbacdd33a39371327),
        (0xb6fa0c8a817bd238, 0x3a68b21edf0d733b),
        (0x36ce1e27351bb092, 0x3a36fc3cd1a4c613),
        (0xb678b7b2ca8c7003, 0xb9d2bb8db2dfce26),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc4081c2a50ad27b, 0x3fc192f2627a74e3),
        (0x3c06268e5916da13, 0xbf60a846a83fecf2),
        (0xbc1be434e30d7a64, 0xbf975eceaabf7f86),
        (0x3bbc2d7f1eb14a8d, 0x3f3617c581be35b1),
        (0xbbb0beeca29a2b7c, 0x3f529934b7a84483),
        (0xbb91b58b66aaaf71, 0xbef18123e875188a),
        (0x3b94740ce7aaa9f5, 0xbefc1f05a2d85165),
        (0xbb3fd873b639f93b, 0x3e9a4e0bc0926545),
        (0xbb233d21389cb3a0, 0x3e98be81ad44d933),
        (0xbabb1f9cab4acd40, 0xbe36f73795e25c01),
        (0x3ababafdeab2d269, 0xbe2c70ab158cd19a),
        (0x3a5796ce7385bd76, 0x3dca262190472d9a),
        (0x3a444e202e2ab843, 0x3db70112aab52bc2),
        (0xb9f5bd722355d797, 0xbd54ec163f88c02f),
        (0x39d5477d64dc86b2, 0xbd3b991575a9daff),
        (0x39314b432a7313b8, 0x3cd8cbabadfd45c5),
        (0xb953683d210d31e2, 0x3cb9867f1ea8526f),
        (0x38f1cbbb05b76a2a, 0xbc56a0c58ce9652e),
        (0x38dd3e4ed9b9011d, 0xbc32c10fa17839d7),
        (0x387159e93c92ec73, 0x3bd06304545a75ab),
        (0x3831184208b9509c, 0x3ba66b5f3f801768),
        (0xb7bf7ef3af29f9f4, 0xbb434a75e18850f7),
        (0xb7b72f867bfa08f9, 0xbb163af2240f11d6),
        (0x374e71e41ce0bd04, 0x3ab2d27f5aad2f87),
        (0xb70e338eb53b4d25, 0x3a82947c379d043b),
        (0xb6a879620ef8dcbc, 0xba1ef0cd1e11d5d6),
        (0xb68cfba8dcf4726a, 0xb9ea897622f547ac),
    ],
    [
        (0x3c68f71f103b6bf8, 0x3fc12f9870d68e18),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc5fd0d25668ad6b, 0xbfb12c11811945f9),
        (0x3bebc70ffc015b97, 0x3f44b638f21f0f76),
        (0x3c1bc10a4e0506fd, 0x3f76d2a897d58353),
        (0x3ba5de76c7b42be1, 0xbf10732e5458ba20),
        (0x3bcfca03362738f1, 0xbf2835929300df3f),
        (0x3b4992f62f113a88, 0x3ec297283816a83b),
        (0xbb50d6dea529ee4f, 0x3ecb73adedf11a43),
        (0xbb0dce80931068a0, 0xbe65b455b903f09c),
        (0x3b029572874fb52c, 0xbe6353f0797a6699),
        (0xba8bf23ef2ca5a38, 0x3dfefc9ac17b1eeb),
        (0xba878bfeeecafab3, 0x3df28535463ce067),
        (0xba096b62f6b59354, 0xbd8dd894cbbee3f9),
        (0xba0f655d10343892, 0xbd79b288cc392b5b),
        (0x3986335c7e33b227, 0x3d14b2ec9ace070c),
        (0xb998edce804ba0e3, 0x3cfaff72f6057832),
        (0xb936e4ec62dfa238, 0xbc95a7bfada02748),
        (0x39060aebd6f74bda, 0xbc7637b7b45ad224),
        (0x38b5e34b545164b0, 0x3c11b216f1fba65d),
        (0x387400818419c16a, 0x3bed6a4aa25cbcad),
        (0x38222d3819e4ad1c, 0xbb873661259c7b49),
        (0xb7c47385cbfe72a2, 0xbb6000a243c0a825),
        (0x3790aa941ed924d2, 0x3af8faeb08a0872d),
        (0x374285ea5fb533a8, 0x3acd2042259aeb3b),
        (0xb70ef65d241a95c5, 0xba66742de4a4a980),
        (0xb6dbeb8e3fb1ee6a, 0xba367fbd4af52810),
        (0xb666550a2bc73761, 0x39d11cea7748dd35),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c60c06e2860e868, 0xbfc0cf3ee98f769b),
        (0xbbfc2c390327e0f7, 0x3f5d26e7af251f79),
        (0xbc394225ed089995, 0x3f965d05948a946a),
        (0xbbcb3460aae30c17, 0xbf335959b8482e40),
        (0xbbf744279fabe84a, 0xbf51cff175d05c2a),
        (0xbb83714c1d2b4ff3, 0x3eeeb59416879106),
        (0xbb89158b4542d2c4, 0x3efaf7544eeac766),
        (0x3b284917e9900e02, 0xbe9720522bb1fc81),
        (0xbb31671693e842ab, 0xbe97c41261705419),
        (0x3a886ab60cf89699, 0x3e343fa0ea5e007d),
        (0xbab8f90fa25a8f71, 0x3e2b5e23abebd7a4),
        (0x3a64e5cdf3a099a5, 0xbdc722397e750312),
        (0xba51768b235cacdf, 0xbdb62f2174e6be49),
        (0xb9fb8faac380f50e, 0x3d5294ab060a163a),
        (0xb9de9b0cd3f3a20f, 0x3d3aace944141c61),
        (0xb96472a1bdce3160, 0xbcd61cc8146cdfa2),
        (0xb9583fcfac500424, 0xbcb8bb3c270a284c),
        (0xb8f56040266e6114, 0x3c54449d96647c28),
        (0xb8c4997de83d7967, 0x3c323712fdd32979),
        (0xb84d0513eb7ddb33, 0xbbcd7cedd05f23ee),
        (0xb8138c0b1312b9f5, 0xbba5d3f2f4d18257),
        (0xb7d6343b2af8488e, 0x3b416f9a8cd62e2d),
        (0xb7b741512b4836ab, 0x3b15b1f1d821a3b4),
        (0x375de94b9017c7ec, 0xbab116d887db5467),
        (0xb720b9ee354d1861, 0xba822c96cd78a74c),
        (0xb6b7416d012e04a4, 0x3a1c3754078e6137),
        (0x3688b4af62f8f0ae, 0x39ea0388f41538a5),
    ],
    [
        (0xbc6742aa46ea9b31, 0xbfc077eede4a0d89),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c3cabc5883fc763, 0x3fb0751548b2924d),
        (0x3bd8c3b817f5bd1a, 0xbf423b5d46a73864),
        (0x3c1e7b0b0210cbce, 0xbf75e2467c8fb832),
        (0xbb9abdc3996174ac, 0x3f0cfe5c189d6e4d),
        (0xbbcf4c0c63a8394c, 0x3f273bbd8c7aef2d),
        (0x3b6d9272577eb6ff, 0xbec06974d3d04287),
        (0xbb56fb637949537f, 0xbeca6081d36e6a2b),
        (0xbaee9b9b25b5c745, 0x3e6334a83cf6092b),
        (0xbb0e3539c19e4313, 0x3e6299571cb4e1fb),
        (0xba80b2cd2f45d38c, 0xbdfb7f5bc0aada50),
        (0xba81831c37e0afc0, 0xbdf1da63b4d6f878),
        (0xba03f6e20cbba970, 0x3d8a9288ceccd9f2),
        (0x3a17055344e86eaa, 0x3d78d1d74c88ee4c),
        (0x39a439e54f23f947, 0xbd127eed3a482d7f),
        (0xb9766b420f9bcda0, 0xbcfa2143706ccb79),
        (0x3933ce1edc9f0854, 0x3c936d1464c4bf11),
        (0x3907d2e3e5d4bfa8, 0x3c758cc3ad07daa3),
        (0xb89bcc0a4f099447, 0xbc0fe0c8ce969c5e),
        (0x38838ca595ab732b, 0xbbec97f785528e08),
        (0x3821507942984ee9, 0x3b84fee6252fe140),
        (0xb7df46f0f7734483, 0x3b5f2dcbce169f1a),
        (0x379086aaae683dab, 0xbaf6b095bc2831d4),
        (0xb76de10fe678bda7, 0xbacc6f522af04f90),
        (0xb703ee192a62f4f4, 0x3a647af3c1c3a557),
        (0x36d76dc7bb35ad50, 0x3a3602a9d860d2a3),
        (0xb66b7ac21c3ed2e5, 0xb9cf58721d5d6786),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c61166b7995967a, 0x3fc0230ba90f2871),
        (0x3bfb746b17280b83, 0xbf59ca16f0c9734e),
        (0xbc375b99fcae0fa4, 0xbf9579c1bdbcfc99),
        (0x3bc1259c7efa1ad5, 0x3f3120ecfac5c017),
        (0xbbd35a76e06cda58, 0x3f511dd26bbe2946),
        (0x3b72ed405d5bc473, 0xbeeb37e7c9a57149),
        (0xbb86e93d7c515f0c, 0xbef9f01e7c1909a0),
        (0xbb2f562d62e60e11, 0x3e94887fe7a88c2c),
        (0xbad010f87629f8e9, 0x3e96e37238841d4b),
        (0xba99b7f19b7b3b20, 0xbe3204b644d6a04b),
        (0xbacc8fcda8a1a29b, 0xbe2a659b13eebf8a),
        (0x3a42b743f24d4d7e, 0x3dc4a40c87601c97),
        (0xba577207717e09cd, 0x3db56f0250e4fcac),
        (0xb9f8225d38237f55, 0xbd50a1011bf4a7dd),
        (0x39d5262f16494ab9, 0xbd39d22a6dc5a57c),
        (0x397bb4017d6e9239, 0x3cd3db0f19da31da),
        (0x392600d1fe84f91f, 0x3cb7fc7d4070961b),
        (0xb8c183aaceabe7cd, 0xbc5243aa7bbc81ca),
        (0x38c5abd4abae47a9, 0xbc31b3c3dd0d783d),
        (0xb86b0ca32f2b28ba, 0x3bcaabdcaba1878d),
        (0xb83fe00611a5322d, 0x3ba541bb402b78f0),
        (0xb7d85111e57bb4b2, 0xbb3fa915cdaddb2d),
        (0xb798d2801af024f8, 0xbb152ba16ad63ba9),
        (0xb7344e83ea6e82fc, 0x3aaf26322e6939a3),
        (0x36eecf18659e6ff5, 0x3a81c51fa64eb3db),
        (0x36ba22c423e879e5, 0xba19d00760b94df4),
        (0xb6523572ecb98917, 0xb9e97bff11d536c1),
    ],
    [
        (0xbc446154ab44acb3, 0x3fbfab0b166d23d8),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4290cc8b89531c, 0xbfafa65c1ce7ebd6),
        (0x3be9bdd7d19e67a3, 0x3f4035bf503ffc1f),
        (0x3be801473cc85f32, 0x3f750d1b04713c41),
        (0x3babe22a416a82d0, 0xbf09cd14a92842a2),
        (0x3bca868e6f4e4828, 0xbf265d504af5d8fe),
        (0x3b5814b5b963313b, 0x3ebd3feeb33d9cee),
        (0xbb4c05f6b4f39a25, 0x3ec96a257062f76e),
        (0x3b0b04fde272af41, 0xbe61254f302b36b2),
        (0x3ad771b26251f405, 0xbe61f11585e04fca),
        (0x3a8b3d53767b22ba, 0x3df89a7674dd98cd),
        (0x3a7b6db02fadeed1, 0x3df13f0ba48c1692),
        (0x3a248f4db1fc6761, 0xbd87d67b47fd268e),
        (0x3a0a84c861831984, 0xbd780381812ad9b0),
        (0xb98981455f0897cf, 0x3d10a41213b1209a),
        (0xb98573618414b59a, 0x3cf9530714f8af8a),
        (0x393f79d60d660532, 0xbc918884a9ff4796),
        (0x3912da336a20734e, 0xbc74ec3ce9f29027),
        (0x3896c9321058ffc6, 0x3c0cde4ce08c6119),
        (0x38894686518435ab, 0x3bebd00bdae20153),
        (0x37ff64113a65f339, 0xbb831454d0bd6f49),
        (0x37e88cc5762134d4, 0xbb5e623854cff634),
        (0x3793b39a55f6f728, 0x3af4b117674c743f),
        (0xb76c32187c451ad3, 0x3acbc2c0ede2286c),
        (0xb7077e3bc9fbf218, 0xba62be3ca56b47a4),
        (0x36dcde9bceca3e29, 0xba35870a752de79f),
        (0x361b91f4f1f1a50f, 0x39ccc9aaaa2d7785),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0x3c50db2c50623ec0, 0xbfbf13fb0c0e6fcd),
        (0x3bf63fd50ee5b7c3, 0x3f5706ed3d935d00),
        (0x3c3a7c91ef9a7da1, 0x3f94af74cbd77bef),
        (0xbbac31e3b8ee4b9e, 0xbf2e9a9e66e5a792),
        (0x3bf35ae9733a6eb6, 0xbf507ec9ed824fcb),
        (0xbb4a1610320898f4, 0x3ee856d4518ab29e),
        (0xbb9af9c37c81d002, 0x3ef9040de830649e),
        (0x3af6ec0adcce1274, 0xbe9262f69c56c652),
        (0xbb34da0db90f5974, 0xbe9618c94a54734d),
        (0x3acfc6bc581d66ff, 0x3e3029d2c8bf70d6),
        (0x3ab60f30da5c2b13, 0x3e2983bca0a3e67c),
        (0x3a60395605657e5e, 0xbdc28e2a00002a82),
        (0xba51c639ac25ee1a, 0xbdb4beeacafc4fad),
        (0x39ed7df744a0b21a, 0x3d4df880e37a5060),
        (0x39d1f98cd4e6ac14, 0x3d3907be43054aec),
        (0xb95bcb238e3393c6, 0xbcd1f10ff685083d),
        (0xb948ad06b5e0f6af, 0xbcb74a1dc71abca4),
        (0xb8f9b8f70559bd9a, 0x3c508d41117c93a5),
        (0xb8b35b059841e08e, 0x3c313795880a7796),
        (0x3865f51a8e31e18f, 0xbbc83ea1cebb3c84),
        (0x3838606560ae8152, 0xbba4b5cc8ded02e2),
        (0xb7df776dd1580636, 0x3b3cdf002471ce45),
        (0x37b71f6e27bbe64e, 0x3b14a97cafad6c19),
        (0x374d40fd5c1285de, 0xbaac7f1793e7126e),
        (0xb6f7da220c547e42, 0xba815f93ab822e98),
        (0xb6a0911cfb822772, 0x3a17b0d93bc7bbc8),
        (0xb6846b70143e3a42, 0x39e8f533a6e89c51),
    ],
    [
        (0xbc5bf01bf61d8ed3, 0xbfbe891b327da16d),
        (0x0000000000000000, 0x0000000000000000),
        (0xbc3636379e4f582e, 0x3fae8533ce07bdb8),
        (0x3bc53b9dd0e187fa, 0xbf3d1253218e31b0),
        (0xbc1eae521e3467c6, 0xbf744e6826476498),
        (0xbb6864ec60e7a724, 0x3f07271a9b5e3cb4),
        (0x3bace83941ef46ea, 0x3f2595b697c8ec04),
        (0xbb5f57ab96615e4a, 0xbeba46b03ecb38cc),
        (0xbb4dc38fda916c56, 0xbec88c173e07621e),
        (0x3add237c79a7a74d, 0x3e5ed9b1754fbd3c),
        (0xbb0d6748ba8d2dd7, 0x3e615891ef314e18),
        (0x3a8caeca005426a6, 0xbdf62ca352cf904e),
        (0x3a7916c3c0550fc0, 0xbdf0b14767c2b010),
        (0x3a25951d764f773e, 0x3d85879bb4a2a8ac),
        (0xb9f96759cfddd4b5, 0x3d7745bbca5b0e82),
        (0xb99c649f85048056, 0xbd0e21736ab8469f),
        (0x399387632a0c4eab, 0xbcf893b256068d4e),
        (0x39225636fb1c995b, 0x3c8fd42c80d0363c),
        (0xb919336e177a135c, 0x3c7455e80611a38d),
        (0xb894f0ece16e6c03, 0xbc0a4682da825109),
        (0x388f20cb0640e817, 0xbbeb12f0bb38aa78),
        (0xb7ff5df786b91715, 0x3b816a7f586e7ed5),
        (0xb7ff8ddcae160d2a, 0x3b5d9fa09d2f4601),
        (0xb7808a35f4fc1e15, 0xbaf2f1ad1590d095),
        (0x373f1307c6b6dcfd, 0xbacb1c0038406c42),
        (0x36f40bb96d08f57f, 0x3a61363c85b6c4db),
        (0x36cb8e8704706c26, 0x3a350e3e866a311e),
        (0xb666cdfdf4125dc0, 0xb9ca841b08aca0a3),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc3b9f1d130797af, 0x3fbe018dac1c17e3),
        (0x3be072789a66cf0c, 0xbf54b994dd05c1fb),
        (0xbc338b4f0d3a4111, 0xbf93f9e0db07e7ef),
        (0xbb9a899561323713, 0x3f2b8e55b75b13ab),
        (0xbbd251fa83bbf98d, 0x3f4fdf68a78bb3d2),
        (0xbb783beff5b1db62, 0xbee5ee9d17106a08),
        (0x3b70935ad53575b8, 0xbef82ee6dfdfedeb),
        (0xbb36eeb8b4341708, 0x3e90962d7f6d61a6),
        (0x3b3ddb699c127f13, 0x3e9560edce7d682d),
        (0xbaadbe6f075c9b8e, 0xbe2d34381d02bdbe),
        (0xbab5faefe806c858, 0xbe28b5a0e74ae0d3),
        (0xba651c26d41110d4, 0x3dc0cab953f05720),
        (0xba4cdfa2ed9741f0, 0x3db41d21c68d7ed3),
        (0x39efbfd30004913c, 0xbd4b2d2dd7cb3c52),
        (0xb9d97fd3cf0000fa, 0xbd384c5369d2e33d),
        (0x3968bd8125247874, 0x3cd04e17ab25ece9),
        (0xb928ec3a9bb7352b, 0x3cb6a38e0aef9402),
        (0x38ed2d1073a2c3f0, 0xbc4e28099fc927bb),
        (0x38d2c69d895d6715, 0xbc30c28e6023fb13),
        (0xb8509810eebec087, 0x3bc62487323559ca),
        (0xb84607b5c3508235, 0x3ba430a24f21f4b6),
        (0x37dcd1d9a6fb187a, 0xbb3a70372a5a35f5),
        (0xb7ba9055f83c1a46, 0xbb142c5d07f39eeb),
        (0x3747d181d0f16b66, 0x3aaa2aaf684b8a30),
        (0x371c65863116b54e, 0x3a80fce718ed7a5e),
        (0xb6b4123df1746b77, 0xba15d07c57e5f8fc),
        (0xb6898be9e1a577c3, 0xb9e870c605b22513),
    ],
    [
        (0xbc4d7a7b8d722118, 0x3fbd84391bb2748d),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c4b439a88ed803c, 0xbfad80edb3c4ea05),
        (0xbbd902d343595b94, 0x3f3a431f5421f7ef),
        (0xbc13a66c5c4dc37b, 0x3f73a282fe7b63a8),
        (0x3bacf6f60b03558d, 0xbf04ed653e607bdd),
        (0xbbb378680e9561c9, 0xbf24e15832bda3c4),
        (0xbb5464c53ca44363, 0x3eb7c5832dd1374f),
        (0x3b40f0848ee7f0af, 0x3ec7c2b80da8df98),
        (0xbadb963cfc74e722, 0xbe5bf160ccd35445),
        (0xbac9dbc6baadaa9d, 0xbe60cda2141b2a1e),
        (0x3a876692ed94eb99, 0x3df41d31d2a419e4),
        (0xba8205fd4d4f295d, 0x3df02f6ca850168a),
        (0xba12525d1774cdfa, 0xbd83901f61054085),
        (0xba17ee02df9f104d, 0xbd7696ce2b748d02),
        (0x39ad0bfbcbf0de66, 0x3d0b6e82997a390e),
        (0x39911978fd395acf, 0x3cf7e214796bec19),
        (0x38f14ad6e06bfc76, 0xbc8d0a3822eede44),
        (0x38ea8d322da40464, 0xbc73c944a2b7aa9b),
        (0x38a984e57b8c632b, 0x3c080748c2fd3ed7),
        (0xb88ca608783691d5, 0x3bea608ebfb61101),
        (0xb7fd3b588ddf9629, 0xbb7fee483d06f76f),
        (0x37f5b0b20d62c293, 0xbb5ce67588df7595),
        (0x37950cdd33b84f9d, 0x3af168f59e870bd0),
        (0x376249d762f468f2, 0x3aca7bdf85923387),
        (0xb6fefd0fec5de258, 0xba5fb7c898747d3e),
        (0x36de35ad136a6653, 0xba349926b4cfc177),
        (0xb6321892fca8dd05, 0x39c87ee43e8319a5),
    ],
    [
        (0x0000000000000000, 0x0000000000000000),
        (0xbc56edd809f4ec43, 0xbfbd09b21e36c0bd),
        (0xbbfd63853cd1251f, 0x3f52c84acfb586b4),
        (0x3c340975d5184af6, 0x3f9355b904fbf7ee),
        (0xbbc2e53f276aee1d, 0xbf28fb570465af0d),
        (0x3bdedf2b372334ad, 0xbf4edc3292ba6cfd),
        (0xbb81b7d6d1c233d4, 0x3ee3e552ee8c2577),
        (0xbb9b6ea46d6db74f, 0x3ef76d44f6a83523),
        (0x3b2fdb93f3ce1be4, 0xbe8e1ee2dc4a3bda),
        (0x3b3cccd4251153ff, 0xbe94b944bbd4c7bf),
        (0xbac468738fc1e577, 0x3e2a8c3a4ce36f16),
        (0x3ac2a4a12f818a06, 0x3e27f8c9ef47068e),
        (0xba40acae788b04d5, 0xbdbe9295749ee471),
        (0x3a1164a101ab66dc, 0xbdb38813c3eb77e4),
        (0x39e5c1a4e6a441fe, 0x3d48c7cda44b7754),
        (0xb9a073f8f8b180ec, 0x3d379e90468589ab),
        (0xb96fed4f7bb5b2dd, 0xbccdca323fb2ed13),
        (0xb95878df442a1f61, 0xbcb6080be2ba756c),
        (0x387f1fa3709d7eff, 0x3c4b9a8e33ba54fc),
        (0xb8dc56a14b4f3362, 0x3c3054766d105243),
        (0xb857137346804b31, 0xbbc44fda8d0857c1),
        (0x384bb0403ea0d8f3, 0xbba3b2575150baf6),
        (0xb7a85fe85251016c, 0x3b384ea306ad146b),
        (0x37b304a7ebd0a72a, 0x3b13b4adf072996e),
        (0x3748b8e00542081f, 0xbaa81d1af5c14291),
        (0x372aac7dc3b524ef, 0xba809dac6fd771cd),
        (0x36baf45596c923ee, 0x3a14269f3ec695f6),
        (0x36759ddc6feb07d0, 0x39e7efc8823b70ff),
    ],
    [
        (0xbc579e384ae35818, 0xbfbc97d79918527d),
        (0x0000000000000000, 0x0000000000000000),
        (0x3c415f3ee939a0c0, 0x3fac95081ab2b511),
        (0x3bd3c58af0675f79, 0xbf37e0b14f7d7c3f),
        (0x3c167e93b7727b21, 0xbf730688f6836a76),
        (0xbb9eaad919470a89, 0x3f030941f6e78e36),
        (0x3bb70db1e3b31015, 0x3f243d5898657a6f),
        (0xbb39f8046976f799, 0xbeb5a39a94f2ad54),
        (0xbb5db47207819abe, 0xbec70b18406146f7),
        (0x3af60534adb9126c, 0x3e597607f9532274),
        (0xbad703287fac9a7c, 0x3e604e788f4f63b9),
        (0xba876fad0fcbbb39, 0xbdf2598919473e7e),
        (0xba640135e9b4a88f, 0xbdef70154273be01),
        (0xba130237cf72ed16, 0x3d81df5c1ca3869c),
        (0x3a1164e3491c1c38, 0x3d75f5256665e918),
        (0x3995f76e145eb882, 0xbd091a176f7295bc),
        (0xb9928726702a883a, 0xbcf73cf989310b2d),
        (0x39202ef4e750363b, 0x3c8a9f0a04e769ab),
        (0x38f077c349f5886f, 0x3c7345b205781db7),
        (0xb8add8bb23d58463, 0xbc0611e4c0bb0b33),
        (0xb87f88c08a95c014, 0xbbe9b88423fdad3a),
        (0xb8109abefbf8e094, 0x3b7d637382cff4b7),
        (0x37f703e341c86f9e, 0x3b5c36bddd0beae2),
        (0xb7802c952716a8be, 0xbaf00eec29705267),
        (0xb755facd187ba820, 0xbac9e2beddaac35c),
        (0xb6e129db12cb1ab9, 0x3a5d51fa8552737e),
        (0xb6cd627e8469503b, 0x3a342847a5dc8d0a),
        (0xb66dc34aae9113ef, 0xb9c6b1f64bc800a0),
    ],
];
