import os
import unittest
from unittest.mock import call, mock_open, patch

from dscan.models.parsers import ReportsParser, TargetOptimization


class TestReportsParsers(unittest.TestCase):

    def test_report_single(self):

        reports_path = os.path.join(os.path.dirname(__file__), 'data')
        results_parser = ReportsParser(reports_path, 'discovery-*.xml')
        values = results_parser.hosts_up()
        self.assertEqual(values, ["172.16.71.132", "172.16.71.133"])

    def test_report_list(self):

        reports_path = os.path.join(os.path.dirname(__file__), 'data')
        results_parser = ReportsParser(reports_path, 'discovery-*.xml')
        values = results_parser.hosts_up()
        self.assertEqual(values, ["172.16.71.132", "172.16.71.133"])

    def test_ipaddress_sort(self):
        ip_list = [
            "255.1.1.1",
            "1.1.1.1",
            "120.10.1.5",
            "120.1.1.5",
            "120.1.1.2",
            "12.30.10.10",
            "12.30.10.1",
            "100.10.0.1",
            "100.9.1.9",
            "10.9.0.11",
            "10.9.1.10",
            "1.1.1.1",
        ]
        expected = [
            call().write("1.1.1.1/32\n"),
            call().write("1.1.1.1/32\n"),
            call().write("10.9.0.11/32\n"),
            call().write("10.9.1.10/32\n"),
            call().write("12.30.10.1/32\n"),
            call().write("12.30.10.10/32\n"),
            call().write("100.9.1.9/32\n"),
            call().write("100.10.0.1/32\n"),
            call().write("120.1.1.2/32\n"),
            call().write("120.1.1.5/32\n"),
            call().write("120.10.1.5/32\n"),
            call().write("255.1.1.1/32\n")
        ]
        mock_obj = mock_open()
        with patch('builtins.open', mock_obj) as mopen:
            reports_path = os.path.join(os.path.dirname(__file__), 'data')
            target_optimize = TargetOptimization(reports_path)
            target_optimize.save(ip_list)
            mopen.assert_has_calls(expected, any_order=True)

    def test_ip_glob_format(self):
        targets = [
            "192.168.10.0",
            "192.168.10.1",
            "192.168.10.2",
            "192.168.10.3",
            "192.168.10.6",
            "192.168.10.7",
            "192.168.10.8",
            "192.168.10.9",
            "192.168.10.10",
            "192.168.10.11",
            "192.168.10.12",
            "192.168.10.14",
            "192.168.10.15",
            "172.16.0.0/16",
            "10.16.0.0",
            "10.16.0.1",
            "10.16.0.2",
            "10.16.0.3",
            "10.16.0.4",
            "10.16.0.5",
            "10.16.0.6",
            "10.16.0.7",
            "10.16.0.8",
            "10.16.0.9",
            "10.16.0.10",
            "10.16.0.11",
            "10.16.0.12",
            "10.16.0.13",
            "10.16.0.14",
            "10.16.0.15",
            "10.16.0.16",
            "10.16.0.17",
            "10.16.0.18",
            "10.16.0.19",
            "10.16.0.20",
            "10.16.0.21",
            "10.16.0.22",
            "10.16.0.23",
            "10.16.0.24",
            "10.16.0.25",
            "10.16.0.26",
            "10.16.0.27",
            "10.16.0.28",
            "10.16.0.29",
            "10.16.0.30",
            "10.16.0.31",
            "10.16.0.32",
            "10.16.0.33",
            "10.16.0.34",
            "10.16.0.35",
            "10.16.0.36",
            "10.16.0.37",
            "10.16.0.38",
            "10.16.0.39",
            "10.16.0.40",
            "10.16.0.41",
            "10.16.0.42",
            "10.16.0.43",
            "10.16.0.44",
            "10.16.0.45",
            "10.16.0.46",
            "10.16.0.47",
            "10.16.0.48",
            "10.16.0.49",
            "10.16.0.50",
            "10.16.0.51",
            "10.16.0.52",
            "10.16.0.53",
            "10.16.0.54",
            "10.16.0.55",
            "10.16.0.56",
            "10.16.0.57",
            "10.16.0.58",
            "10.16.0.59",
            "10.16.0.60",
            "10.16.0.61",
            "10.16.0.62",
            "10.16.0.63",
            "10.16.0.64",
            "10.16.0.65",
            "10.16.0.66",
            "10.16.0.67",
            "10.16.0.68",
            "10.16.0.69",
            "10.16.0.70",
            "10.16.0.71",
            "10.16.0.72",
            "10.16.0.73",
            "10.16.0.74",
            "10.16.0.75",
            "10.16.0.76",
            "10.16.0.77",
            "10.16.0.78",
            "10.16.0.79",
            "10.16.0.80",
            "10.16.0.81",
            "10.16.0.82",
            "10.16.0.83",
            "10.16.0.84",
            "10.16.0.85",
            "10.16.0.86",
            "10.16.0.87",
            "10.16.0.88",
            "10.16.0.89",
            "10.16.0.90",
            "10.16.0.91",
            "10.16.0.92",
            "10.16.0.93",
            "10.16.0.94",
            "10.16.0.95",
            "10.16.0.96",
            "10.16.0.97",
            "10.16.0.98",
            "10.16.0.99",
            "10.16.0.100",
            "10.16.0.101",
            "10.16.0.102",
            "10.16.0.103",
            "10.16.0.104",
            "10.16.0.105",
            "10.16.0.106",
            "10.16.0.107",
            "10.16.0.108",
            "10.16.0.109",
            "10.16.0.110",
            "10.16.0.111",
            "10.16.0.112",
            "10.16.0.113",
            "10.16.0.114",
            "10.16.0.115",
            "10.16.0.116",
            "10.16.0.117",
            "10.16.0.118",
            "10.16.0.119",
            "10.16.0.120",
            "10.16.0.121",
            "10.16.0.122",
            "10.16.0.123",
            "10.16.0.124",
            "10.16.0.125",
            "10.16.0.126",
            "10.16.0.127",
            "10.16.0.128",
            "10.16.0.129",
            "10.16.0.130",
            "10.16.0.131",
            "10.16.0.132",
            "10.16.0.133",
            "10.16.0.134",
            "10.16.0.135",
            "10.16.0.136",
            "10.16.0.137",
            "10.16.0.138",
            "10.16.0.139",
            "10.16.0.140",
            "10.16.0.141",
            "10.16.0.142",
            "10.16.0.143",
            "10.16.0.144",
            "10.16.0.145",
            "10.16.0.146",
            "10.16.0.147",
            "10.16.0.148",
            "10.16.0.149",
            "10.16.0.150",
            "10.16.0.151",
            "10.16.0.152",
            "10.16.0.153",
            "10.16.0.154",
            "10.16.0.155",
            "10.16.0.156",
            "10.16.0.157",
            "10.16.0.158",
            "10.16.0.159",
            "10.16.0.160",
            "10.16.0.161",
            "10.16.0.162",
            "10.16.0.163",
            "10.16.0.164",
            "10.16.0.165",
            "10.16.0.166",
            "10.16.0.167",
            "10.16.0.168",
            "10.16.0.169",
            "10.16.0.170",
            "10.16.0.171",
            "10.16.0.172",
            "10.16.0.173",
            "10.16.0.174",
            "10.16.0.175",
            "10.16.0.176",
            "10.16.0.177",
            "10.16.0.178",
            "10.16.0.179",
            "10.16.0.180",
            "10.16.0.181",
            "10.16.0.182",
            "10.16.0.183",
            "10.16.0.184",
            "10.16.0.185",
            "10.16.0.186",
            "10.16.0.187",
            "10.16.0.188",
            "10.16.0.189",
            "10.16.0.190",
            "10.16.0.191",
            "10.16.0.192",
            "10.16.0.193",
            "10.16.0.194",
            "10.16.0.195",
            "10.16.0.196",
            "10.16.0.197",
            "10.16.0.198",
            "10.16.0.199",
            "10.16.0.200",
            "10.16.0.201",
            "10.16.0.202",
            "10.16.0.203",
            "10.16.0.204",
            "10.16.0.205",
            "10.16.0.206",
            "10.16.0.207",
            "10.16.0.208",
            "10.16.0.209",
            "10.16.0.210",
            "10.16.0.211",
            "10.16.0.212",
            "10.16.0.213",
            "10.16.0.214",
            "10.16.0.215",
            "10.16.0.216",
            "10.16.0.217",
            "10.16.0.218",
            "10.16.0.219",
            "10.16.0.220",
            "10.16.0.221",
            "10.16.0.222",
            "10.16.0.223",
            "10.16.0.224",
            "10.16.0.225",
            "10.16.0.226",
            "10.16.0.227",
            "10.16.0.228",
            "10.16.0.229",
            "10.16.0.230",
            "10.16.0.231",
            "10.16.0.232",
            "10.16.0.233",
            "10.16.0.234",
            "10.16.0.235",
            "10.16.0.236",
            "10.16.0.237",
            "10.16.0.238",
            "10.16.0.239",
            "10.16.0.240",
            "10.16.0.241",
            "10.16.0.242",
            "10.16.0.243",
            "10.16.0.244",
            "10.16.0.245",
            "10.16.0.246",
            "10.16.0.247",
            "10.16.0.248",
            "10.16.0.249",
            "10.16.0.250",
            "10.16.0.251",
            "10.16.0.252",
            "10.16.0.253",
            "10.16.0.254",
            "10.16.0.255",
            "10.15.0.0",
            "10.15.0.1",
            "10.15.0.2",
            "10.15.0.3",
            "10.15.0.4",
            "10.15.0.5",
            "10.15.0.6",
            "10.15.0.7",
            "10.15.0.8",
            "10.15.0.9",
            "10.15.0.10",
            "10.15.0.11",
            "10.15.0.12",
            "10.15.0.13",
            "10.15.0.14",
            "10.15.0.15",
            "10.15.0.16",
            "10.15.0.17",
            "10.15.0.18",
            "10.15.0.19",
            "10.15.0.20",
            "10.15.0.21",
            "10.15.0.22",
            "10.15.0.23",
            "10.15.0.24",
            "10.15.0.25",
            "10.15.0.26",
            "10.15.0.27",
            "10.15.0.28",
            "10.15.0.29",
            "10.15.0.30",
            "10.15.0.31",
            "10.15.0.32",
            "10.15.0.33",
            "10.15.0.34",
            "10.15.0.35",
            "10.15.0.36",
            "10.15.0.37",
            "10.15.0.38",
            "10.15.0.39",
            "10.15.0.40",
            "10.15.0.41",
            "10.15.0.42",
            "10.15.0.43",
            "10.15.0.44",
            "10.15.0.45",
            "10.15.0.46",
            "10.15.0.47",
            "10.15.0.48",
            "10.15.0.49",
            "10.15.0.50",
            "10.15.0.51",
            "10.15.0.52",
            "10.15.0.53",
            "10.15.0.54",
            "10.15.0.55",
            "10.15.0.56",
            "10.15.0.57",
            "10.15.0.58",
            "10.15.0.59",
            "10.15.0.60",
            "10.15.0.61",
            "10.15.0.62",
            "10.15.0.63",
            "10.15.0.64",
            "10.15.0.65",
            "10.15.0.66",
            "10.15.0.67",
            "10.15.0.68",
            "10.15.0.69",
            "10.15.0.70",
            "10.15.0.71",
            "10.15.0.72",
            "10.15.0.73",
            "10.15.0.74",
            "10.15.0.75",
            "10.15.0.76",
            "10.15.0.77",
            "10.15.0.78",
            "10.15.0.79",
            "10.15.0.80",
            "10.15.0.81",
            "10.15.0.82",
            "10.15.0.83",
            "10.15.0.84",
            "10.15.0.85",
            "10.15.0.86",
            "10.15.0.87",
            "10.15.0.88",
            "10.15.0.89",
            "10.15.0.90",
            "10.15.0.91",
            "10.15.0.92",
            "10.15.0.93",
            "10.15.0.94",
            "10.15.0.95",
            "10.15.0.96",
            "10.15.0.97",
            "10.15.0.98",
            "10.15.0.99",
            "10.15.0.100",
            "10.15.0.101",
            "10.15.0.102",
            "10.15.0.103",
            "10.15.0.104",
            "10.15.0.105",
            "10.15.0.106",
            "10.15.0.107",
            "10.15.0.108",
            "10.15.0.109",
            "10.15.0.110",
            "10.15.0.111",
            "10.15.0.112",
            "10.15.0.113",
            "10.15.0.114",
            "10.15.0.115",
            "10.15.0.116",
            "10.15.0.117",
            "10.15.0.118",
            "10.15.0.119",
            "10.15.0.120",
            "10.15.0.121",
            "10.15.0.122",
            "10.15.0.123",
            "10.15.0.124",
            "10.15.0.125",
            "10.15.0.126",
            "10.15.0.127",
            "10.15.0.128",
            "10.15.0.129",
            "10.15.0.130",
            "10.15.0.131",
            "10.15.0.132",
            "10.15.0.133",
            "10.15.0.134",
            "10.15.0.135",
            "10.15.0.136",
            "10.15.0.137",
            "10.15.0.138",
            "10.15.0.139",
            "10.15.0.140",
            "10.15.0.141",
            "10.15.0.142",
            "10.15.0.143",
            "10.15.0.144",
            "10.15.0.145",
            "10.15.0.146",
            "10.15.0.147",
            "10.15.0.148",
            "10.15.0.149",
            "10.15.0.150",
            "10.15.0.151",
            "10.15.0.152",
            "10.15.0.153",
            "10.15.0.154",
            "10.15.0.155",
            "10.15.0.156",
            "10.15.0.157",
            "10.15.0.158",
            "10.15.0.159",
            "10.15.0.160",
            "10.15.0.161",
            "10.15.0.162",
            "10.15.0.163",
            "10.15.0.164",
            "10.15.0.165",
            "10.15.0.166",
            "10.15.0.167",
            "10.15.0.168",
            "10.15.0.169",
            "10.15.0.170",
            "10.15.0.171",
            "10.15.0.172",
            "10.15.0.173",
            "10.15.0.174",
            "10.15.0.175",
            "10.15.0.176",
            "10.15.0.177",
            "10.15.0.178",
            "10.15.0.179",
            "10.15.0.180",
            "10.15.0.181",
            "10.15.0.182",
            "10.15.0.183",
            "10.15.0.184",
            "10.15.0.185",
            "10.15.0.186",
            "10.15.0.187",
            "10.15.0.188",
            "10.15.0.189",
            "10.15.0.190",
            "10.15.0.191",
            "10.15.0.192",
            "10.15.0.193",
            "10.15.0.194",
            "10.15.0.195",
            "10.15.0.196",
            "10.15.0.197",
            "10.15.0.198",
            "10.15.0.199",
            "10.15.0.200",
            "10.15.0.201",
            "10.15.0.202",
            "10.15.0.203",
            "10.15.0.204",
            "10.15.0.205",
            "10.15.0.206",
            "10.15.0.207",
            "10.15.0.208",
            "10.15.0.209",
            "10.15.0.210",
            "10.15.0.211",
            "10.15.0.212",
            "10.15.0.213",
            "10.15.0.214",
            "10.15.0.215",
            "10.15.0.216",
            "10.15.0.217",
            "10.15.0.218",
            "10.15.0.219",
            "10.15.0.220",
            "10.15.0.221",
            "10.15.0.222",
            "10.15.0.223",
            "10.15.0.224",
            "10.15.0.225",
            "10.15.0.226",
            "10.15.0.227",
            "10.15.0.228",
            "10.15.0.229",
            "10.15.0.230",
            "10.15.0.231",
            "10.15.0.232",
            "10.15.0.233",
            "10.15.0.234",
            "10.15.0.235",
            "10.15.0.236",
            "10.15.0.237",
            "10.15.0.238",
            "10.15.0.239",
            "10.15.0.240",
            "10.15.0.241",
            "10.15.0.242",
            "10.15.0.243",
            "10.15.0.244",
            "10.15.0.245",
            "10.15.0.246",
            "10.15.0.247",
            "10.15.0.248",
            "10.15.0.249",
            "10.15.0.250",
            "10.15.0.251",
            "10.15.0.252",
            "10.15.0.253",
            "10.15.0.254",
            "10.15.0.255"
        ]
        reports_path = "fake/live-targets.work"
        expected_write = [
            call(reports_path, "wt"),
            call().write('10.15.0.0/24\n'),
            call().write('10.16.0.0/24\n'),
            call().write('192.168.10.0/30\n'),
            call().write('192.168.10.6-12\n'),
            call().write('192.168.10.14/31\n')
        ]

        mock_obj = mock_open()
        with patch('builtins.open', mock_obj) as mopen:
            target_optimize = TargetOptimization(reports_path)
            target_optimize.save(targets)
            mock_obj().writelines.assert_called_once()
            mopen.assert_has_calls(expected_write, any_order=True)


if __name__ == '__main__':
    unittest.main()
