/*
 * rcracki_mt is a multithreaded implementation and fork of the original 
 * RainbowCrack
 *
 * Copyright 2009, 2010 Danil Niggebrugge <niggebrugge@fox-it.com>
 * Copyright 2009, 2010, 2011, 2012 James Nobis <quel@quelrod.net>
 *
 * This file is part of rcracki_mt.
 *
 * rcracki_mt is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * rcracki_mt is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with rcracki_mt.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "lm2ntlm.h"

LM2NTLMcorrector::LM2NTLMcorrector()
{
	progressCurrentCombination = 0;
	totalCurrentCombination = 1;
	counterOverall = 0;
	countCombinations = 0;
	countTotalCombinations = 0;
	counter = 0;
	fillMapW();
	aborting = false;
	sBinary = "";

}

std::string LM2NTLMcorrector::getBinary()
{
	return sBinary;
}

bool LM2NTLMcorrector::LMPasswordCorrectUnicode( std::string hexPassword, unsigned char* pNTLMHash, std::string& sNTLMPassword ) //, unsigned char* pLMPassword
{
	std::string sPlain = "";

	uint32_t i;
	for (i = 0; i < hexPassword.size() / 2; i++)
	{
		std::string sSub = hexPassword.substr(i * 2, 2);
		int nValue;
		sscanf(sSub.c_str(), "%02x", &nValue);
		sPlain += (unsigned char)nValue;
	}

	memcpy(NTLMHash, pNTLMHash, MD4_DIGEST_LENGTH);


	unsigned long int tmpLength = sPlain.size() * 2;
	unsigned char* pLMPassword = new unsigned char[tmpLength];

	//printf("Searching for unicode password.\n");
	//printf("NTLM hash:          %s\n\n", sNTLMHash.c_str());
	
	setvbuf(stdout, NULL, _IONBF,0);

	gettimeofday( &tvStart, NULL );
	previousClock = clock();

#ifndef _WIN32
	tty_init();
#endif

	if (startCorrecting(sPlain, sNTLMPassword, pLMPassword))
	{
		sBinary = ByteToStr(pLMPassword, tmpLength).c_str();
		//printf("\nFound unicode password: %s\n", sNTLMPassword.c_str());
		//printf("Password in hex: %s\n", sBinary.c_str());
		writeEndStats();
#ifndef _WIN32
		tty_done();
#endif
		return true;
	}
	else
	{
		//printf("\ncase correction for password %s fail!\n", sPlain.c_str());
		writeEndStats();
#ifndef _WIN32
		tty_done();
#endif
		return false;
	}
}

bool LM2NTLMcorrector::startCorrecting( std::string sLMPassword, std::string& sNTLMPassword, unsigned char* pLMPassword )
{
	if (sLMPassword.size() == 0)
	{
		sNTLMPassword = "";
		return true;
	}

	std::string muteMe = sLMPassword;
	int length = muteMe.size();

	unsigned char* pMuteMe = new unsigned char[length];
	unsigned char* pTempMute = new unsigned char[length * 2];

	int i;
	for (i = 0; i < length; i++)
	{
		pMuteMe[i] = muteMe[i];
		pTempMute[i * 2    ] = muteMe[i];
		pTempMute[i * 2 + 1] = 0x00;
		unsigned char muteChar = pMuteMe[i];
		int sizeMapForChar = m_mapChar[muteChar].size();
		int j;
		for (j = 0; j < sizeMapForChar; j++)
		{
			currentCharmap[i][j] = m_mapChar[muteChar][j];
		}
	}
	
	int* jAtPos = new int[length];
	int* sizeAtPos = new int[length];
	bool* fullAtPos = new bool[length];

	int setSize;
	for (setSize = 0; setSize <= length; setSize++)
	{
		int cntFull = 0;

		// clear all 'fullatpos' before new setSize
		int i;
		for (i=0; i < length; i++)
		{
			fullAtPos[i] = false;
		}

		//printf("Trying full unicode map for %d/%d characters...\t\t\n", setSize, length);
		printf("Trying full unicode map for %d/%d characters...%-20s\n", setSize, length, "");

		bool notFirst = true;

		// start at end and set 'full' combination
		countCombinations = 0;
		countTotalCombinations = calculateTotalCombinations(length, setSize);

		int sPos = length - 1;
		while (sPos >= 0 && notFirst) // finding combinations for current 'setSize'
		{
			if (aborting)
				return false;

			if (cntFull < setSize)
			{
				if (fullAtPos[sPos] == false)
				{
					fullAtPos[sPos] = true;
					cntFull++;
				}
				sPos--;
			}
			else
			{
				if (fullAtPos[sPos] == false && setSize > 0)
				{
					fullAtPos[sPos] = true;
					cntFull++;
					
					// reset positions after sPos
					int k;
					for (k = sPos+1; k < length; k++)
					{
						if (fullAtPos[k] == true)
						{
							fullAtPos[k] = false;
							cntFull--;
						}
					}
					// start at end again
					sPos = length - 1;
				}
				else
				{
					sPos--;
				}
			}
			// we have a combination
			if (cntFull == setSize)
			{
				countCombinations++;

				setupCombinationAtPositions(length, pMuteMe, pTempMute, jAtPos, fullAtPos, sizeAtPos);

				if (checkPermutations(length, pTempMute, jAtPos, sizeAtPos, pLMPassword, sNTLMPassword))
				{
					return true;
				}
			}

			if (setSize == 0)
				notFirst = false;
		}
	}
	return false;
}

// set up combination at positions
void LM2NTLMcorrector::setupCombinationAtPositions(int length, unsigned char* pMuteMe, unsigned char* pTempMute, int* jAtPos, bool* fullAtPos, int* sizeAtPos)
{
	progressCurrentCombination = 0;
	totalCurrentCombination = 1;

	int i;
	for (i=0; i < length; i++)
	{
		pTempMute[i*2] = currentCharmap[i][0];
		pTempMute[i*2+1] = currentCharmap[i][1]; // reset to first char in map
	
		jAtPos[i] = 0; // reset charcounter for this char (that is all chars)

		// based on combination, set full map or only upper/lowercase
		if (fullAtPos[i] == true)
		{
			unsigned char muteChar = pMuteMe[i];
			long unsigned int sizeMapForChar = m_mapChar[muteChar].size()/2; // 2 bytes per char
			sizeAtPos[i] = sizeMapForChar;
		}
		else
		{
			sizeAtPos[i] = 2;
		}

		totalCurrentCombination *= sizeAtPos[i];
	}
	//printf("Trying %"PRIu64" passwords for current combination\t\t\r", totalCurrentCombination);
}

// go check all permutations for this combination
bool LM2NTLMcorrector::checkPermutations( int length, unsigned char* pTempMute, int* jAtPos, int* sizeAtPos, unsigned char* pLMPassword, std::string& sNTLMPassword )
{
	int pos = length - 1;

	while (pos >= 0)
	{
		counter++;

		pos = length - 1;

		int jAtCurPos = jAtPos[pos];
		int sizeMapForCharPos = sizeAtPos[pos];
		// move to start of string and find character with permutations left
		while (jAtCurPos >= sizeMapForCharPos-1 && pos >= -1)
		{
			pos--;
			if (pos >= 0 )
			{
				jAtCurPos = jAtPos[pos];
				sizeMapForCharPos = sizeAtPos[pos];
			}
		}
		if (pos < 0)
			continue;

		// next permutation for character
		jAtCurPos++;
		jAtPos[pos] = jAtCurPos;
		
		pTempMute[pos*2] = currentCharmap[pos][jAtCurPos*2];
		pTempMute[pos*2+1] = currentCharmap[pos][jAtCurPos*2+1];
		
		// reset positions after pos
		int k;
		for (k = pos+1; k < length; k++)
		{
			jAtPos[k] = 0;
			pTempMute[k*2] = currentCharmap[k][0]; // reset to first char in map
			pTempMute[k*2+1] = currentCharmap[k][1];
		}

		if (checkNTLMPassword(pTempMute, length, sNTLMPassword) == true)
		{
			int i;
			for (i = 0; i < length*2; i++)
				pLMPassword[i] = pTempMute[i];
			return true;
		}

		if (counter > 10000) // don't check clocks too often
		{
			gettimeofday( &tvEnd, NULL );
	
			tvFinal = sub_timeofday( tvEnd, tvStart );

			clock_t currentClock = clock();
			float fTime = 1.0f * (currentClock - previousClock);
			if (fTime > 1.0f * CLOCKS_PER_SEC)
			{
				float progressPercentageCurrentCombination = progressCurrentCombination * 100.0f / totalCurrentCombination;
				float fTime = 1.0f * tvFinal.tv_sec + 1.0f * tvFinal.tv_usec / 1000000;
				float currentSpeed = (counterOverall + progressCurrentCombination) / fTime / 1000000;

				//printf("%.2f%% of combination %d/%d (%.2f Mhashes/s)\t\t\t\t\r", progressPercentageCurrentCombination, countCombinations, countTotalCombinations, currentSpeed);
				printf("%.2f%% of combination %d/%d (%.2f Mhashes/s)%-30s\r", progressPercentageCurrentCombination, countCombinations, countTotalCombinations, currentSpeed, "");

				previousClock = clock();
				#ifdef _WIN32
				if (_kbhit())
				{
					int ch = _getch();
					ch = toupper(ch);
					if (ch == 'S')
					{
						aborting = true;
						printf( "\nAborting unicode correction for this hash...\n");
					}
					else
					{
						printf( "\nPress 'S' to skip unicode correction for this hash...\n");
					}
				}
				#else
				int c = tty_getchar();
				if (c >= 0) {
					tty_flush();
					if (c==115) { // = s
						aborting = true;
						printf( "\nAborting unicode correction for this hash...\n");
					}
					else {
						printf( "\nPress 's' to skip unicode correction for this hash...\n");
					}
				}
				#endif
				if (aborting)
					return false;
			}
			counter = 0;
		}
		
		progressCurrentCombination++;
		counterOverall++;
	}
	return false;
}

// check password, maybe integrate this function in checkPermutations() for performance reasons.
bool LM2NTLMcorrector::checkNTLMPassword( unsigned char* pLMPassword, int nLMPasswordLen, std::string& sNTLMPassword )
{
	unsigned char md[MD4_DIGEST_LENGTH];

	MD4_NEW( pLMPassword, nLMPasswordLen * 2, md );

	if (memcmp(md, NTLMHash, MD4_DIGEST_LENGTH) == 0)
	{
		sNTLMPassword = "";
		int i;
		for (i = 0; i < nLMPasswordLen; i++) {
			sNTLMPassword += char(pLMPassword[i * 2]);
		}
		return true;
	}
	else
		return false;
}

void LM2NTLMcorrector::checkAbort()
{
#ifdef _WIN32
	if (_kbhit())
	{
		int ch = _getch();
		ch = toupper(ch);
		if (ch == 'S')
		{
			aborting = true;
			printf( "\nAborting unicode correction for this hash...\n");
		}
		else
		{
			printf( "\nPress 'S' to skip unicode correction for this hash...\n");
		}
	}
#endif
}

void LM2NTLMcorrector::writeEndStats()
{
	gettimeofday( &tvEnd, NULL );
	
	tvFinal = sub_timeofday( tvEnd, tvStart );

	float fTime = 1.0f * tvFinal.tv_sec + 1.0f * tvFinal.tv_usec / 1000000;
	float speedOverall = counterOverall / fTime / 1000000;
	printf("\nTried %s passwords in %.2f s (%.2f Mhashes/s)\n", uint64tostr(counterOverall).c_str(), fTime, speedOverall);

	printf("\n");
}

int LM2NTLMcorrector::calculateTotalCombinations(int length, int setSize)
{
	return factorial(length) / (factorial(setSize) * factorial(length-setSize));
}

uint64_t LM2NTLMcorrector::factorial (int num)
{
	uint64_t result = 1;

	while ( num > 1 )
	{
		result *= num;
		num--;
	}

	return result;
}

// convert some bytes into a string
std::string LM2NTLMcorrector::ByteToStr(const unsigned char* pData, int nLen)
{
	std::string sRet = "";
	int i;
	for (i = 0; i < nLen/2; i++)
	{
		char szByte[3];
		sprintf(szByte, "%02x", pData[i*2+1]); // swap 2-byte characters again
		sRet += szByte;
		sprintf(szByte, "%02x", pData[i*2]);
		sRet += szByte;
	}

	return sRet;
}

void LM2NTLMcorrector::addToMapW(unsigned char key, unsigned char value1, unsigned char value2)
{
	unsigned long int cnt = m_mapChar[key].size();
	m_mapChar[key][cnt] = value2;
	m_mapChar[key][cnt+1] = value1; //reverse for endiannes
}

// construct the mappings, would be nicer in a separate (importable) file
void LM2NTLMcorrector::fillMapW()
{
	addToMapW(0x01, 0x00, 0x01);
	addToMapW(0x01, 0x26, 0x3A);
	addToMapW(0x02, 0x00, 0x02);
	addToMapW(0x02, 0x26, 0x3B);
	addToMapW(0x03, 0x00, 0x03);
	addToMapW(0x03, 0x26, 0x65);
	addToMapW(0x04, 0x00, 0x04);
	addToMapW(0x04, 0x26, 0x66);
	addToMapW(0x05, 0x00, 0x05);
	addToMapW(0x05, 0x26, 0x63);
	addToMapW(0x06, 0x00, 0x06);
	addToMapW(0x06, 0x26, 0x60);
	addToMapW(0x07, 0x00, 0x07);
	addToMapW(0x07, 0x00, 0xB7);
	addToMapW(0x07, 0x20, 0x22);
	addToMapW(0x07, 0x20, 0x24);
	addToMapW(0x07, 0x20, 0x26);
	addToMapW(0x07, 0x22, 0x19);
	addToMapW(0x07, 0x22, 0xC5);
	addToMapW(0x07, 0x30, 0xFB);
	addToMapW(0x08, 0x00, 0x08);
	addToMapW(0x08, 0x25, 0xD8);
	addToMapW(0x09, 0x00, 0x09);
	addToMapW(0x09, 0x20, 0xDD);
	addToMapW(0x09, 0x25, 0xCB);
	addToMapW(0x09, 0x30, 0x07);
	addToMapW(0x0a, 0x00, 0x0A);
	addToMapW(0x0a, 0x25, 0xD9);
	addToMapW(0x0b, 0x00, 0x0B);
	addToMapW(0x0b, 0x26, 0x42);
	addToMapW(0x0c, 0x00, 0x0C);
	addToMapW(0x0c, 0x26, 0x40);
	addToMapW(0x0d, 0x00, 0x0D);
	addToMapW(0x0d, 0x26, 0x6A);
	addToMapW(0x0e, 0x00, 0x0E);
	addToMapW(0x0e, 0x26, 0x6B);
	addToMapW(0x0f, 0x00, 0x0F);
	addToMapW(0x0f, 0x00, 0xA4);
	addToMapW(0x0f, 0x26, 0x3C);
	addToMapW(0x10, 0x00, 0x10);
	addToMapW(0x10, 0x25, 0xBA);
	addToMapW(0x11, 0x00, 0x11);
	addToMapW(0x11, 0x25, 0xC4);
	addToMapW(0x12, 0x00, 0x12);
	addToMapW(0x12, 0x21, 0x95);
	addToMapW(0x13, 0x00, 0x13);
	addToMapW(0x13, 0x20, 0x3C);
	addToMapW(0x14, 0x00, 0x14);
	addToMapW(0x14, 0x00, 0xB6);
	addToMapW(0x15, 0x00, 0x15);
	addToMapW(0x15, 0x00, 0xA7);
	addToMapW(0x16, 0x00, 0x16);
	addToMapW(0x16, 0x02, 0xC9);
	addToMapW(0x16, 0x25, 0xAC);
	addToMapW(0x17, 0x00, 0x17);
	addToMapW(0x17, 0x21, 0xA8);
	addToMapW(0x18, 0x00, 0x18);
	addToMapW(0x18, 0x21, 0x91);
	addToMapW(0x19, 0x00, 0x19);
	addToMapW(0x19, 0x21, 0x93);
	addToMapW(0x1a, 0x00, 0x1A);
	addToMapW(0x1a, 0x21, 0x92);
	addToMapW(0x1b, 0x00, 0x1B);
	addToMapW(0x1b, 0x21, 0x90);
	addToMapW(0x1c, 0x00, 0x1C);
	addToMapW(0x1c, 0x22, 0x1F);
	addToMapW(0x1d, 0x00, 0x1D);
	addToMapW(0x1d, 0x21, 0x94);
	addToMapW(0x1e, 0x00, 0x1E);
	addToMapW(0x1e, 0x25, 0xB2);
	addToMapW(0x1f, 0x00, 0x1F);
	addToMapW(0x1f, 0x25, 0xBC);
	addToMapW(0x20, 0x00, 0x20);
	addToMapW(0x20, 0x20, 0x00);
	addToMapW(0x20, 0x20, 0x01);
	addToMapW(0x20, 0x20, 0x02);
	addToMapW(0x20, 0x20, 0x03);
	addToMapW(0x20, 0x20, 0x04);
	addToMapW(0x20, 0x20, 0x05);
	addToMapW(0x20, 0x20, 0x06);
	addToMapW(0x20, 0x30, 0x00);
	addToMapW(0x21, 0x00, 0x21);
	addToMapW(0x21, 0x00, 0xA1);
	addToMapW(0x21, 0x01, 0xC3);
	addToMapW(0x21, 0xFF, 0x01);
	addToMapW(0x22, 0x00, 0x22);
	addToMapW(0x22, 0x00, 0xA8);
	addToMapW(0x22, 0x02, 0xBA);
	addToMapW(0x22, 0x03, 0x08);
	addToMapW(0x22, 0x03, 0x0E);
	addToMapW(0x22, 0x20, 0x1C);
	addToMapW(0x22, 0x20, 0x1D);
	addToMapW(0x22, 0x20, 0x1E);
	addToMapW(0x22, 0x20, 0x33);
	addToMapW(0x22, 0x20, 0x35);
	addToMapW(0x22, 0x27, 0x5D);
	addToMapW(0x22, 0x27, 0x5E);
	addToMapW(0x22, 0x30, 0x1D);
	addToMapW(0x22, 0x30, 0x1E);
	addToMapW(0x22, 0x30, 0x1F);
	addToMapW(0x22, 0xFF, 0x02);
	addToMapW(0x23, 0x00, 0x23);
	addToMapW(0x23, 0xFF, 0x03);
	addToMapW(0x24, 0x00, 0x24);
	addToMapW(0x24, 0xFF, 0x04);
	addToMapW(0x25, 0x00, 0x25);
	addToMapW(0x25, 0x06, 0x6A);
	addToMapW(0x25, 0x20, 0x30);
	addToMapW(0x25, 0xFF, 0x05);
	addToMapW(0x26, 0x00, 0x26);
	addToMapW(0x26, 0xFF, 0x06);
	addToMapW(0x27, 0x00, 0x27);
	addToMapW(0x27, 0x00, 0xB4);
	addToMapW(0x27, 0x02, 0xB9);
	addToMapW(0x27, 0x02, 0xBB);
	addToMapW(0x27, 0x02, 0xBC);
	addToMapW(0x27, 0x02, 0xC8);
	addToMapW(0x27, 0x02, 0xCA);
	addToMapW(0x27, 0x02, 0xCB);
	addToMapW(0x27, 0x03, 0x00);
	addToMapW(0x27, 0x03, 0x01);
	addToMapW(0x27, 0x20, 0x18);
	addToMapW(0x27, 0x20, 0x19);
	addToMapW(0x27, 0x20, 0x1A);
	addToMapW(0x27, 0x20, 0x32);
	addToMapW(0x27, 0x27, 0x5B);
	addToMapW(0x27, 0x27, 0x5C);
	addToMapW(0x27, 0xFF, 0x07);
	addToMapW(0x28, 0x00, 0x28);
	addToMapW(0x28, 0x23, 0x20);
	addToMapW(0x28, 0xFF, 0x08);
	addToMapW(0x29, 0x00, 0x29);
	addToMapW(0x29, 0x23, 0x21);
	addToMapW(0x29, 0xFF, 0x09);
	addToMapW(0x2a, 0x00, 0x2A);
	addToMapW(0x2a, 0x22, 0x17);
	addToMapW(0x2a, 0xFF, 0x0A);
	addToMapW(0x2b, 0x00, 0x2B);
	addToMapW(0x2b, 0x00, 0xB1);
	addToMapW(0x2b, 0x20, 0x20);
	addToMapW(0x2b, 0x20, 0x21);
	addToMapW(0x2b, 0xFF, 0x0B);
	addToMapW(0x2c, 0x00, 0x2C);
	addToMapW(0x2c, 0x00, 0xB8);
	addToMapW(0x2c, 0x03, 0x27);
	addToMapW(0x2c, 0x20, 0x1A);
	addToMapW(0x2c, 0x20, 0x1E);
	addToMapW(0x2c, 0xFF, 0x0C);
	addToMapW(0x2d, 0x00, 0x2D);
	addToMapW(0x2d, 0x00, 0xAC);
	addToMapW(0x2d, 0x00, 0xAD);
	addToMapW(0x2d, 0x20, 0x10);
	addToMapW(0x2d, 0x20, 0x11);
	addToMapW(0x2d, 0x20, 0x13);
	addToMapW(0x2d, 0x20, 0x14);
	addToMapW(0x2d, 0x22, 0x12);
	addToMapW(0x2d, 0x22, 0x13);
	addToMapW(0x2d, 0xFF, 0x0D);
	addToMapW(0x2e, 0x00, 0x2E);
	addToMapW(0x2e, 0x20, 0x26);
	addToMapW(0x2e, 0xFF, 0x0E);
	addToMapW(0x2f, 0x00, 0x2F);
	addToMapW(0x2f, 0x20, 0x44);
	addToMapW(0x2f, 0x22, 0x15);
	addToMapW(0x2f, 0x22, 0x16);
	addToMapW(0x2f, 0xFF, 0x0F);
	addToMapW(0x30, 0x00, 0x30);
	addToMapW(0x30, 0x20, 0x70);
	addToMapW(0x30, 0x20, 0x80);
	addToMapW(0x30, 0xFF, 0x10);
	addToMapW(0x31, 0x00, 0x31);
	addToMapW(0x31, 0x00, 0xB9);
	addToMapW(0x31, 0x00, 0xBC);
	addToMapW(0x31, 0x00, 0xBD);
	addToMapW(0x31, 0x20, 0x81);
	addToMapW(0x31, 0xFF, 0x11);
	addToMapW(0x32, 0x00, 0x32);
	addToMapW(0x32, 0x00, 0xB2);
	addToMapW(0x32, 0x20, 0x82);
	addToMapW(0x32, 0xFF, 0x12);
	addToMapW(0x33, 0x00, 0x33);
	addToMapW(0x33, 0x00, 0xB3);
	addToMapW(0x33, 0x00, 0xBE);
	addToMapW(0x33, 0x20, 0x83);
	addToMapW(0x33, 0xFF, 0x13);
	addToMapW(0x34, 0x00, 0x34);
	addToMapW(0x34, 0x20, 0x74);
	addToMapW(0x34, 0x20, 0x84);
	addToMapW(0x34, 0xFF, 0x14);
	addToMapW(0x35, 0x00, 0x35);
	addToMapW(0x35, 0x20, 0x75);
	addToMapW(0x35, 0x20, 0x85);
	addToMapW(0x35, 0xFF, 0x15);
	addToMapW(0x36, 0x00, 0x36);
	addToMapW(0x36, 0x20, 0x76);
	addToMapW(0x36, 0x20, 0x86);
	addToMapW(0x36, 0xFF, 0x16);
	addToMapW(0x37, 0x00, 0x37);
	addToMapW(0x37, 0x20, 0x77);
	addToMapW(0x37, 0x20, 0x87);
	addToMapW(0x37, 0xFF, 0x17);
	addToMapW(0x38, 0x00, 0x38);
	addToMapW(0x38, 0x20, 0x78);
	addToMapW(0x38, 0x20, 0x88);
	addToMapW(0x38, 0x22, 0x1E);
	addToMapW(0x38, 0xFF, 0x18);
	addToMapW(0x39, 0x00, 0x39);
	addToMapW(0x39, 0x20, 0x78);
	addToMapW(0x39, 0x20, 0x89);
	addToMapW(0x39, 0xFF, 0x19);
	addToMapW(0x3a, 0x00, 0x3A);
	addToMapW(0x3a, 0x05, 0x89);
	addToMapW(0x3a, 0x20, 0x26);
	addToMapW(0x3a, 0x22, 0x36);
	addToMapW(0x3a, 0xFF, 0x1A);
	addToMapW(0x3b, 0x00, 0x3B);
	addToMapW(0x3b, 0x03, 0x7E);
	addToMapW(0x3b, 0xFF, 0x1B);
	addToMapW(0x3c, 0x00, 0x3C);
	addToMapW(0x3c, 0x00, 0xAB);
	addToMapW(0x3c, 0x20, 0x39);
	addToMapW(0x3c, 0x23, 0x29);
	addToMapW(0x3c, 0x30, 0x08);
	addToMapW(0x3c, 0xFF, 0x1C);
	addToMapW(0x3d, 0x00, 0x3D);
	addToMapW(0x3d, 0x22, 0x61);
	addToMapW(0x3d, 0x22, 0x64);
	addToMapW(0x3d, 0x22, 0x65);
	addToMapW(0x3d, 0xFF, 0x1D);
	addToMapW(0x3e, 0x00, 0x3E);
	addToMapW(0x3e, 0x00, 0xBB);
	addToMapW(0x3e, 0x20, 0x3A);
	addToMapW(0x3e, 0x23, 0x2A);
	addToMapW(0x3e, 0x30, 0x09);
	addToMapW(0x3e, 0xFF, 0x1E);
	addToMapW(0x3f, 0x00, 0x3F);
	addToMapW(0x40, 0x00, 0x40);
	addToMapW(0x40, 0xFF, 0x20);
	addToMapW(0x41, 0x00, 0x41);
	addToMapW(0x41, 0x00, 0x61);
	addToMapW(0x41, 0x00, 0xAA);
	addToMapW(0x41, 0x00, 0xC0);
	addToMapW(0x41, 0x00, 0xC1);
	addToMapW(0x41, 0x00, 0xC2);
	addToMapW(0x41, 0x00, 0xC3);
	addToMapW(0x41, 0x00, 0xC4);
	addToMapW(0x41, 0x00, 0xC5);
	addToMapW(0x41, 0x00, 0xC6);
	addToMapW(0x41, 0x00, 0xE0);
	addToMapW(0x41, 0x00, 0xE1);
	addToMapW(0x41, 0x00, 0xE2);
	addToMapW(0x41, 0x00, 0xE3);
	addToMapW(0x41, 0x00, 0xE4);
	addToMapW(0x41, 0x00, 0xE5);
	addToMapW(0x41, 0x00, 0xE6);
	addToMapW(0x41, 0x01, 0x00);
	addToMapW(0x41, 0x01, 0x01);
	addToMapW(0x41, 0x01, 0x02);
	addToMapW(0x41, 0x01, 0x03);
	addToMapW(0x41, 0x01, 0x04);
	addToMapW(0x41, 0x01, 0x05);
	addToMapW(0x41, 0x01, 0xCD);
	addToMapW(0x41, 0x01, 0xCE);
	addToMapW(0x41, 0x01, 0xDE);
	addToMapW(0x41, 0x01, 0xDF);
	addToMapW(0x41, 0x03, 0xB1);
	addToMapW(0x41, 0x21, 0x2B);
	addToMapW(0x41, 0xFF, 0x21);
	addToMapW(0x41, 0xFF, 0x41);
	addToMapW(0x42, 0x00, 0x42);
	addToMapW(0x42, 0x00, 0x62);
	addToMapW(0x42, 0x01, 0x80);
	addToMapW(0x42, 0x21, 0x2C);
	addToMapW(0x42, 0xFF, 0x22);
	addToMapW(0x42, 0xFF, 0x42);
	addToMapW(0x43, 0x00, 0x43);
	addToMapW(0x43, 0x00, 0x63);
	addToMapW(0x43, 0x00, 0xA2);
	addToMapW(0x43, 0x00, 0xA9);
	addToMapW(0x43, 0x00, 0xC7);
	addToMapW(0x43, 0x00, 0xE7);
	addToMapW(0x43, 0x00, 0xE8);
	addToMapW(0x43, 0x01, 0x06);
	addToMapW(0x43, 0x01, 0x07);
	addToMapW(0x43, 0x01, 0x08);
	addToMapW(0x43, 0x01, 0x09);
	addToMapW(0x43, 0x01, 0x0A);
	addToMapW(0x43, 0x01, 0x0B);
	addToMapW(0x43, 0x01, 0x0C);
	addToMapW(0x43, 0x01, 0x0D);
	addToMapW(0x43, 0x21, 0x02);
	addToMapW(0x43, 0x21, 0x2D);
	addToMapW(0x43, 0xFF, 0x23);
	addToMapW(0x43, 0xFF, 0x43);
	addToMapW(0x44, 0x00, 0x44);
	addToMapW(0x44, 0x00, 0x64);
	addToMapW(0x44, 0x00, 0xD0);
	addToMapW(0x44, 0x00, 0xF0);
	addToMapW(0x44, 0x01, 0x0E);
	addToMapW(0x44, 0x01, 0x0F);
	addToMapW(0x44, 0x01, 0x10);
	addToMapW(0x44, 0x01, 0x11);
	addToMapW(0x44, 0x01, 0x89);
	addToMapW(0x44, 0x03, 0xB4);
	addToMapW(0x44, 0x26, 0x6A);
	addToMapW(0x44, 0x26, 0x6B);
	addToMapW(0x44, 0xFF, 0x24);
	addToMapW(0x44, 0xFF, 0x44);
	addToMapW(0x45, 0x00, 0x45);
	addToMapW(0x45, 0x00, 0x65);
	addToMapW(0x45, 0x00, 0xC8);
	addToMapW(0x45, 0x00, 0xC9);
	addToMapW(0x45, 0x00, 0xCA);
	addToMapW(0x45, 0x00, 0xCB);
	addToMapW(0x45, 0x00, 0xE8);
	addToMapW(0x45, 0x00, 0xE9);
	addToMapW(0x45, 0x00, 0xEA);
	addToMapW(0x45, 0x00, 0xEB);
	addToMapW(0x45, 0x01, 0x12);
	addToMapW(0x45, 0x01, 0x13);
	addToMapW(0x45, 0x01, 0x14);
	addToMapW(0x45, 0x01, 0x15);
	addToMapW(0x45, 0x01, 0x16);
	addToMapW(0x45, 0x01, 0x17);
	addToMapW(0x45, 0x01, 0x18);
	addToMapW(0x45, 0x01, 0x19);
	addToMapW(0x45, 0x01, 0x1A);
	addToMapW(0x45, 0x01, 0x1B);
	addToMapW(0x45, 0x03, 0xB5);
	addToMapW(0x45, 0x21, 0x07);
	addToMapW(0x45, 0x21, 0x2E);
	addToMapW(0x45, 0x21, 0x2F);
	addToMapW(0x45, 0x21, 0x30);
	addToMapW(0x45, 0xFF, 0x25);
	addToMapW(0x45, 0xFF, 0x45);
	addToMapW(0x46, 0x00, 0x46);
	addToMapW(0x46, 0x00, 0x66);
	addToMapW(0x46, 0x01, 0x91);
	addToMapW(0x46, 0x01, 0x92);
	addToMapW(0x46, 0x03, 0xA6);
	addToMapW(0x46, 0x03, 0xC6);
	addToMapW(0x46, 0x21, 0x31);
	addToMapW(0x46, 0xFF, 0x26);
	addToMapW(0x46, 0xFF, 0x46);
	addToMapW(0x47, 0x00, 0x47);
	addToMapW(0x47, 0x00, 0x67);
	addToMapW(0x47, 0x01, 0x1C);
	addToMapW(0x47, 0x01, 0x1D);
	addToMapW(0x47, 0x01, 0x1E);
	addToMapW(0x47, 0x01, 0x1F);
	addToMapW(0x47, 0x01, 0x20);
	addToMapW(0x47, 0x01, 0x21);
	addToMapW(0x47, 0x01, 0x22);
	addToMapW(0x47, 0x01, 0x23);
	addToMapW(0x47, 0x01, 0xE4);
	addToMapW(0x47, 0x01, 0xE5);
	addToMapW(0x47, 0x01, 0xE6);
	addToMapW(0x47, 0x01, 0xE7);
	addToMapW(0x47, 0x02, 0x61);
	addToMapW(0x47, 0x03, 0x93);
	addToMapW(0x47, 0x21, 0x0A);
	addToMapW(0x47, 0xFF, 0x27);
	addToMapW(0x47, 0xFF, 0x47);
	addToMapW(0x48, 0x00, 0x48);
	addToMapW(0x48, 0x00, 0x68);
	addToMapW(0x48, 0x01, 0x24);
	addToMapW(0x48, 0x01, 0x25);
	addToMapW(0x48, 0x01, 0x26);
	addToMapW(0x48, 0x01, 0x27);
	addToMapW(0x48, 0x04, 0xBB);
	addToMapW(0x48, 0x21, 0x0B);
	addToMapW(0x48, 0x21, 0x0C);
	addToMapW(0x48, 0x21, 0x0D);
	addToMapW(0x48, 0x21, 0x0E);
	addToMapW(0x48, 0xFF, 0x28);
	addToMapW(0x48, 0xFF, 0x48);
	addToMapW(0x49, 0x00, 0x49);
	addToMapW(0x49, 0x00, 0x69);
	addToMapW(0x49, 0x00, 0xCC);
	addToMapW(0x49, 0x00, 0xCD);
	addToMapW(0x49, 0x00, 0xCE);
	addToMapW(0x49, 0x00, 0xCF);
	addToMapW(0x49, 0x00, 0xEC);
	addToMapW(0x49, 0x00, 0xED);
	addToMapW(0x49, 0x00, 0xEE);
	addToMapW(0x49, 0x00, 0xEF);
	addToMapW(0x49, 0x01, 0x28);
	addToMapW(0x49, 0x01, 0x29);
	addToMapW(0x49, 0x01, 0x2A);
	addToMapW(0x49, 0x01, 0x2B);
	addToMapW(0x49, 0x01, 0x2C);
	addToMapW(0x49, 0x01, 0x2D);
	addToMapW(0x49, 0x01, 0x2E);
	addToMapW(0x49, 0x01, 0x2F);
	addToMapW(0x49, 0x01, 0x30);
	addToMapW(0x49, 0x01, 0x31);
	addToMapW(0x49, 0x01, 0x97);
	addToMapW(0x49, 0x01, 0xCF);
	addToMapW(0x49, 0x01, 0xD0);
	addToMapW(0x49, 0x21, 0x10);
	addToMapW(0x49, 0x21, 0x11);
	addToMapW(0x49, 0xFF, 0x29);
	addToMapW(0x49, 0xFF, 0x49);
	addToMapW(0x4a, 0x00, 0x4A);
	addToMapW(0x4a, 0x00, 0x6A);
	addToMapW(0x4a, 0x01, 0x34);
	addToMapW(0x4a, 0x01, 0x35);
	addToMapW(0x4a, 0x01, 0xF0);
	addToMapW(0x4a, 0xFF, 0x2A);
	addToMapW(0x4a, 0xFF, 0x4A);
	addToMapW(0x4b, 0x00, 0x4B);
	addToMapW(0x4b, 0x00, 0x6B);
	addToMapW(0x4b, 0x01, 0x36);
	addToMapW(0x4b, 0x01, 0x37);
	addToMapW(0x4b, 0x01, 0xE8);
	addToMapW(0x4b, 0x01, 0xE9);
	addToMapW(0x4b, 0x21, 0x2A);
	addToMapW(0x4b, 0xFF, 0x2B);
	addToMapW(0x4b, 0xFF, 0x4B);
	addToMapW(0x4c, 0x00, 0x4C);
	addToMapW(0x4c, 0x00, 0x6C);
	addToMapW(0x4c, 0x00, 0xA3);
	addToMapW(0x4c, 0x01, 0x39);
	addToMapW(0x4c, 0x01, 0x3A);
	addToMapW(0x4c, 0x01, 0x3B);
	addToMapW(0x4c, 0x01, 0x3C);
	addToMapW(0x4c, 0x01, 0x3D);
	addToMapW(0x4c, 0x01, 0x3E);
	addToMapW(0x4c, 0x01, 0x41);
	addToMapW(0x4c, 0x01, 0x42);
	addToMapW(0x4c, 0x01, 0x9A);
	addToMapW(0x4c, 0x20, 0xA4);
	addToMapW(0x4c, 0x21, 0x12);
	addToMapW(0x4c, 0x21, 0x13);
	addToMapW(0x4c, 0xFF, 0x2C);
	addToMapW(0x4c, 0xFF, 0x4C);
	addToMapW(0x4d, 0x00, 0x4D);
	addToMapW(0x4d, 0x00, 0x6D);
	addToMapW(0x4d, 0x21, 0x33);
	addToMapW(0x4d, 0xFF, 0x2D);
	addToMapW(0x4d, 0xFF, 0x4D);
	addToMapW(0x4e, 0x00, 0x4E);
	addToMapW(0x4e, 0x00, 0x6E);
	addToMapW(0x4e, 0x00, 0xD1);
	addToMapW(0x4e, 0x00, 0xF1);
	addToMapW(0x4e, 0x01, 0x43);
	addToMapW(0x4e, 0x01, 0x44);
	addToMapW(0x4e, 0x01, 0x45);
	addToMapW(0x4e, 0x01, 0x46);
	addToMapW(0x4e, 0x01, 0x47);
	addToMapW(0x4e, 0x01, 0x48);
	addToMapW(0x4e, 0x20, 0x7F);
	addToMapW(0x4e, 0x21, 0x15);
	addToMapW(0x4e, 0x22, 0x29);
	addToMapW(0x4e, 0xFF, 0x2E);
	addToMapW(0x4e, 0xFF, 0x4E);
	addToMapW(0x4f, 0x00, 0x4F);
	addToMapW(0x4f, 0x00, 0x6F);
	addToMapW(0x4f, 0x00, 0xB0);
	addToMapW(0x4f, 0x00, 0xBA);
	addToMapW(0x4f, 0x00, 0xD2);
	addToMapW(0x4f, 0x00, 0xD3);
	addToMapW(0x4f, 0x00, 0xD4);
	addToMapW(0x4f, 0x00, 0xD5);
	addToMapW(0x4f, 0x00, 0xD6);
	addToMapW(0x4f, 0x00, 0xD8);
	addToMapW(0x4f, 0x00, 0xF2);
	addToMapW(0x4f, 0x00, 0xF3);
	addToMapW(0x4f, 0x00, 0xF4);
	addToMapW(0x4f, 0x00, 0xF5);
	addToMapW(0x4f, 0x00, 0xF6);
	addToMapW(0x4f, 0x00, 0xF8);
	addToMapW(0x4f, 0x01, 0x4C);
	addToMapW(0x4f, 0x01, 0x4D);
	addToMapW(0x4f, 0x01, 0x4E);
	addToMapW(0x4f, 0x01, 0x4F);
	addToMapW(0x4f, 0x01, 0x50);
	addToMapW(0x4f, 0x01, 0x51);
	addToMapW(0x4f, 0x01, 0x52);
	addToMapW(0x4f, 0x01, 0x53);
	addToMapW(0x4f, 0x01, 0x9F);
	addToMapW(0x4f, 0x01, 0xA0);
	addToMapW(0x4f, 0x01, 0xA1);
	addToMapW(0x4f, 0x01, 0xD1);
	addToMapW(0x4f, 0x01, 0xD2);
	addToMapW(0x4f, 0x01, 0xEA);
	addToMapW(0x4f, 0x01, 0xEB);
	addToMapW(0x4f, 0x01, 0xEC);
	addToMapW(0x4f, 0x01, 0xED);
	addToMapW(0x4f, 0x03, 0xA9);
	addToMapW(0x4f, 0x20, 0xDD);
	addToMapW(0x4f, 0x21, 0x26);
	addToMapW(0x4f, 0x21, 0x34);
	addToMapW(0x4f, 0x22, 0x05);
	addToMapW(0x4f, 0x30, 0x07);
	addToMapW(0x4f, 0xFF, 0x2F);
	addToMapW(0x4f, 0xFF, 0x4F);
	addToMapW(0x50, 0x00, 0x50);
	addToMapW(0x50, 0x00, 0x70);
	addToMapW(0x50, 0x03, 0xC0);
	addToMapW(0x50, 0x20, 0xA7);
	addToMapW(0x50, 0x21, 0x18);
	addToMapW(0x50, 0x21, 0x19);
	addToMapW(0x50, 0xFF, 0x30);
	addToMapW(0x50, 0xFF, 0x50);
	addToMapW(0x51, 0x00, 0x51);
	addToMapW(0x51, 0x00, 0x71);
	addToMapW(0x51, 0x21, 0x1A);
	addToMapW(0x51, 0xFF, 0x31);
	addToMapW(0x51, 0xFF, 0x51);
	addToMapW(0x52, 0x00, 0x52);
	addToMapW(0x52, 0x00, 0x72);
	addToMapW(0x52, 0x00, 0xAE);
	addToMapW(0x52, 0x01, 0x54);
	addToMapW(0x52, 0x01, 0x55);
	addToMapW(0x52, 0x01, 0x56);
	addToMapW(0x52, 0x01, 0x57);
	addToMapW(0x52, 0x01, 0x58);
	addToMapW(0x52, 0x01, 0x59);
	addToMapW(0x52, 0x21, 0x1B);
	addToMapW(0x52, 0x21, 0x1C);
	addToMapW(0x52, 0x21, 0x1D);
	addToMapW(0x52, 0xFF, 0x32);
	addToMapW(0x52, 0xFF, 0x52);
	addToMapW(0x53, 0x00, 0x53);
	addToMapW(0x53, 0x00, 0x73);
	addToMapW(0x53, 0x00, 0xDF);
	addToMapW(0x53, 0x01, 0x5A);
	addToMapW(0x53, 0x01, 0x5B);
	addToMapW(0x53, 0x01, 0x5C);
	addToMapW(0x53, 0x01, 0x5D);
	addToMapW(0x53, 0x01, 0x5E);
	addToMapW(0x53, 0x01, 0x5F);
	addToMapW(0x53, 0x01, 0x60);
	addToMapW(0x53, 0x01, 0x61);
	addToMapW(0x53, 0x01, 0xA9);
	addToMapW(0x53, 0x03, 0xA3);
	addToMapW(0x53, 0x03, 0xC3);
	addToMapW(0x53, 0x22, 0x11);
	addToMapW(0x53, 0xFF, 0x33);
	addToMapW(0x53, 0xFF, 0x53);
	addToMapW(0x54, 0x00, 0x54);
	addToMapW(0x54, 0x00, 0x74);
	addToMapW(0x54, 0x00, 0xDE);
	addToMapW(0x54, 0x00, 0xFE);
	addToMapW(0x54, 0x01, 0x62);
	addToMapW(0x54, 0x01, 0x63);
	addToMapW(0x54, 0x01, 0x64);
	addToMapW(0x54, 0x01, 0x65);
	addToMapW(0x54, 0x01, 0x66);
	addToMapW(0x54, 0x01, 0x67);
	addToMapW(0x54, 0x01, 0xAB);
	addToMapW(0x54, 0x01, 0xAE);
	addToMapW(0x54, 0x03, 0xC4);
	addToMapW(0x54, 0x21, 0x22);
	addToMapW(0x54, 0xFF, 0x34);
	addToMapW(0x54, 0xFF, 0x54);
	addToMapW(0x55, 0x00, 0x55);
	addToMapW(0x55, 0x00, 0x75);
	addToMapW(0x55, 0x00, 0xB5);
	addToMapW(0x55, 0x00, 0xD9);
	addToMapW(0x55, 0x00, 0xDA);
	addToMapW(0x55, 0x00, 0xDB);
	addToMapW(0x55, 0x00, 0xDC);
	addToMapW(0x55, 0x00, 0xF9);
	addToMapW(0x55, 0x00, 0xFA);
	addToMapW(0x55, 0x00, 0xFB);
	addToMapW(0x55, 0x00, 0xFC);
	addToMapW(0x55, 0x01, 0x68);
	addToMapW(0x55, 0x01, 0x69);
	addToMapW(0x55, 0x01, 0x6A);
	addToMapW(0x55, 0x01, 0x6B);
	addToMapW(0x55, 0x01, 0x6C);
	addToMapW(0x55, 0x01, 0x6D);
	addToMapW(0x55, 0x01, 0x6E);
	addToMapW(0x55, 0x01, 0x6F);
	addToMapW(0x55, 0x01, 0x70);
	addToMapW(0x55, 0x01, 0x71);
	addToMapW(0x55, 0x01, 0x72);
	addToMapW(0x55, 0x01, 0x73);
	addToMapW(0x55, 0x01, 0xAF);
	addToMapW(0x55, 0x01, 0xB0);
	addToMapW(0x55, 0x01, 0xD3);
	addToMapW(0x55, 0x01, 0xD4);
	addToMapW(0x55, 0x01, 0xD5);
	addToMapW(0x55, 0x01, 0xD6);
	addToMapW(0x55, 0x01, 0xD7);
	addToMapW(0x55, 0x01, 0xD8);
	addToMapW(0x55, 0x01, 0xD9);
	addToMapW(0x55, 0x01, 0xDA);
	addToMapW(0x55, 0x01, 0xDB);
	addToMapW(0x55, 0x01, 0xDC);
	addToMapW(0x55, 0x03, 0xBC);
	addToMapW(0x55, 0xFF, 0x35);
	addToMapW(0x55, 0xFF, 0x55);
	addToMapW(0x56, 0x00, 0x56);
	addToMapW(0x56, 0x00, 0x76);
	addToMapW(0x56, 0x22, 0x1A);
	addToMapW(0x56, 0x27, 0x13);
	addToMapW(0x56, 0xFF, 0x36);
	addToMapW(0x56, 0xFF, 0x56);
	addToMapW(0x57, 0x00, 0x57);
	addToMapW(0x57, 0x00, 0x77);
	addToMapW(0x57, 0x01, 0x74);
	addToMapW(0x57, 0x01, 0x75);
	addToMapW(0x57, 0xFF, 0x37);
	addToMapW(0x57, 0xFF, 0x57);
	addToMapW(0x58, 0x00, 0x58);
	addToMapW(0x58, 0x00, 0x78);
	addToMapW(0x58, 0x00, 0xD7);
	addToMapW(0x58, 0xFF, 0x38);
	addToMapW(0x58, 0xFF, 0x58);
	addToMapW(0x59, 0x00, 0x59);
	addToMapW(0x59, 0x00, 0x79);
	addToMapW(0x59, 0x00, 0xA5);
	addToMapW(0x59, 0x00, 0xDD);
	addToMapW(0x59, 0x00, 0xFD);
	addToMapW(0x59, 0x00, 0xFF);
	addToMapW(0x59, 0x01, 0x76);
	addToMapW(0x59, 0x01, 0x77);
	addToMapW(0x59, 0x01, 0x78);
	addToMapW(0x59, 0xFF, 0x39);
	addToMapW(0x59, 0xFF, 0x59);
	addToMapW(0x5a, 0x00, 0x5A);
	addToMapW(0x5a, 0x00, 0x7A);
	addToMapW(0x5a, 0x01, 0x79);
	addToMapW(0x5a, 0x01, 0x7A);
	addToMapW(0x5a, 0x01, 0x7B);
	addToMapW(0x5a, 0x01, 0x7C);
	addToMapW(0x5a, 0x01, 0x7D);
	addToMapW(0x5a, 0x01, 0x7E);
	addToMapW(0x5a, 0x01, 0xB6);
	addToMapW(0x5a, 0x21, 0x24);
	addToMapW(0x5a, 0x21, 0x28);
	addToMapW(0x5a, 0xFF, 0x3A);
	addToMapW(0x5a, 0xFF, 0x5A);
	addToMapW(0x5b, 0x00, 0x5B);
	addToMapW(0x5b, 0x30, 0x1A);
	addToMapW(0x5b, 0xFF, 0x3B);
	addToMapW(0x5c, 0x00, 0x5C);
	addToMapW(0x5c, 0x00, 0xA5);
	addToMapW(0x5c, 0x22, 0x16);
	addToMapW(0x5c, 0xFF, 0x3C);
	addToMapW(0x5d, 0x00, 0x5D);
	addToMapW(0x5d, 0x30, 0x1B);
	addToMapW(0x5d, 0xFF, 0x3D);
	addToMapW(0x5e, 0x00, 0x5E);
	addToMapW(0x5e, 0x02, 0xC4);
	addToMapW(0x5e, 0x02, 0xC6);
	addToMapW(0x5e, 0x02, 0xC7);
	addToMapW(0x5e, 0x02, 0xD8);
	addToMapW(0x5e, 0x03, 0x02);
	addToMapW(0x5e, 0x03, 0x06);
	addToMapW(0x5e, 0x03, 0x0C);
	addToMapW(0x5e, 0x23, 0x03);
	addToMapW(0x5e, 0xFF, 0x3E);
	addToMapW(0x5f, 0x00, 0x5F);
	addToMapW(0x5f, 0x00, 0xAF);
	addToMapW(0x5f, 0x00, 0xBE);
	addToMapW(0x5f, 0x00, 0xDE);
	addToMapW(0x5f, 0x00, 0xFE);
	addToMapW(0x5f, 0x02, 0xCD);
	addToMapW(0x5f, 0x03, 0x31);
	addToMapW(0x5f, 0x03, 0x32);
	addToMapW(0x5f, 0x20, 0x17);
	addToMapW(0x5f, 0x30, 0xFC);
	addToMapW(0x5f, 0xFF, 0x3F);
	addToMapW(0x60, 0x00, 0x60);
	addToMapW(0x60, 0x02, 0xCB);
	addToMapW(0x60, 0x03, 0x00);
	addToMapW(0x60, 0x20, 0x18);
	addToMapW(0x60, 0x20, 0x35);
	addToMapW(0x60, 0xFF, 0x40);
	addToMapW(0x7b, 0x00, 0x7B);
	addToMapW(0x7b, 0xFF, 0x5B);
	addToMapW(0x7c, 0x00, 0x7C);
	addToMapW(0x7c, 0x00, 0xA6);
	addToMapW(0x7c, 0x01, 0xC0);
	addToMapW(0x7c, 0x22, 0x23);
	addToMapW(0x7c, 0x27, 0x58);
	addToMapW(0x7c, 0xFF, 0x5C);
	addToMapW(0x7d, 0x00, 0x7D);
	addToMapW(0x7d, 0x30, 0x1B);
	addToMapW(0x7d, 0xFF, 0x5D);
	addToMapW(0x7e, 0x00, 0x7E);
	addToMapW(0x7e, 0x02, 0xDC);
	addToMapW(0x7e, 0x03, 0x03);
	addToMapW(0x7e, 0x22, 0x3C);
	addToMapW(0x7e, 0x22, 0x48);
	addToMapW(0x7e, 0xFF, 0x5E);
	addToMapW(0x7f, 0x00, 0x7F);
	addToMapW(0x7f, 0x23, 0x02);
	addToMapW(0x7f, 0x26, 0x60);
	addToMapW(0x7f, 0x26, 0x63);
	addToMapW(0x7f, 0x26, 0x65);
	addToMapW(0x7f, 0x26, 0x66);
	addToMapW(0x80, 0x00, 0x80);
	addToMapW(0x80, 0x00, 0xC7);
	addToMapW(0x80, 0x00, 0xE7);
	addToMapW(0x80, 0x01, 0x06);
	addToMapW(0x80, 0x01, 0x07);
	addToMapW(0x80, 0x03, 0x91);
	addToMapW(0x80, 0x03, 0xB1);
	addToMapW(0x80, 0x04, 0x10);
	addToMapW(0x80, 0x04, 0x30);
	addToMapW(0x80, 0x05, 0xD0);
	addToMapW(0x80, 0x20, 0xAC);
	addToMapW(0x81, 0x00, 0x81);
	addToMapW(0x81, 0x03, 0x92);
	addToMapW(0x81, 0x03, 0xB2);
	addToMapW(0x81, 0x04, 0x02);
	addToMapW(0x81, 0x04, 0x11);
	addToMapW(0x81, 0x04, 0x31);
	addToMapW(0x81, 0x04, 0x52);
	addToMapW(0x81, 0x05, 0xD1);
	addToMapW(0x82, 0x00, 0x82);
	addToMapW(0x82, 0x03, 0x93);
	addToMapW(0x82, 0x03, 0xB3);
	addToMapW(0x82, 0x04, 0x12);
	addToMapW(0x82, 0x04, 0x32);
	addToMapW(0x82, 0x05, 0xD2);
	addToMapW(0x82, 0x20, 0x1A);
	addToMapW(0x83, 0x00, 0x83);
	addToMapW(0x83, 0x03, 0x94);
	addToMapW(0x83, 0x03, 0xB4);
	addToMapW(0x83, 0x04, 0x03);
	addToMapW(0x83, 0x04, 0x13);
	addToMapW(0x83, 0x04, 0x33);
	addToMapW(0x83, 0x04, 0x53);
	addToMapW(0x83, 0x05, 0xD3);
	addToMapW(0x84, 0x00, 0x84);
	addToMapW(0x84, 0x03, 0x95);
	addToMapW(0x84, 0x03, 0xB5);
	addToMapW(0x84, 0x04, 0x14);
	addToMapW(0x84, 0x04, 0x34);
	addToMapW(0x84, 0x05, 0xD4);
	addToMapW(0x84, 0x20, 0x1E);
	addToMapW(0x85, 0x03, 0x96);
	addToMapW(0x85, 0x03, 0xB6);
	addToMapW(0x85, 0x04, 0x01);
	addToMapW(0x85, 0x04, 0x15);
	addToMapW(0x85, 0x04, 0x35);
	addToMapW(0x85, 0x04, 0x51);
	addToMapW(0x85, 0x05, 0xD5);
	addToMapW(0x85, 0x20, 0x26);
	addToMapW(0x86, 0x00, 0x86);
	addToMapW(0x86, 0x03, 0x97);
	addToMapW(0x86, 0x03, 0xB7);
	addToMapW(0x86, 0x04, 0x16);
	addToMapW(0x86, 0x04, 0x36);
	addToMapW(0x86, 0x05, 0xD6);
	addToMapW(0x86, 0x20, 0x20);
	addToMapW(0x87, 0x00, 0x87);
	addToMapW(0x87, 0x03, 0x98);
	addToMapW(0x87, 0x03, 0xB8);
	addToMapW(0x87, 0x04, 0x04);
	addToMapW(0x87, 0x04, 0x17);
	addToMapW(0x87, 0x04, 0x37);
	addToMapW(0x87, 0x04, 0x54);
	addToMapW(0x87, 0x05, 0xD7);
	addToMapW(0x87, 0x20, 0x21);
	addToMapW(0x88, 0x00, 0x88);
	addToMapW(0x88, 0x02, 0xC6);
	addToMapW(0x88, 0x03, 0x99);
	addToMapW(0x88, 0x03, 0xB9);
	addToMapW(0x88, 0x04, 0x18);
	addToMapW(0x88, 0x04, 0x38);
	addToMapW(0x88, 0x05, 0xD8);
	addToMapW(0x89, 0x00, 0x89);
	addToMapW(0x89, 0x03, 0x9A);
	addToMapW(0x89, 0x03, 0xBA);
	addToMapW(0x89, 0x04, 0x05);
	addToMapW(0x89, 0x04, 0x19);
	addToMapW(0x89, 0x04, 0x39);
	addToMapW(0x89, 0x04, 0x55);
	addToMapW(0x89, 0x05, 0xD9);
	addToMapW(0x89, 0x20, 0x30);
	addToMapW(0x8a, 0x00, 0x8A);
	addToMapW(0x8a, 0x01, 0x50);
	addToMapW(0x8a, 0x01, 0x51);
	addToMapW(0x8a, 0x01, 0x56);
	addToMapW(0x8a, 0x01, 0x57);
	addToMapW(0x8a, 0x03, 0x9B);
	addToMapW(0x8a, 0x03, 0xBB);
	addToMapW(0x8a, 0x04, 0x1A);
	addToMapW(0x8a, 0x04, 0x3A);
	addToMapW(0x8a, 0x05, 0xDA);
	addToMapW(0x8b, 0x00, 0x8B);
	addToMapW(0x8b, 0x03, 0x9C);
	addToMapW(0x8b, 0x03, 0xBC);
	addToMapW(0x8b, 0x04, 0x06);
	addToMapW(0x8b, 0x04, 0x1B);
	addToMapW(0x8b, 0x04, 0x3B);
	addToMapW(0x8b, 0x04, 0x56);
	addToMapW(0x8b, 0x05, 0xDB);
	addToMapW(0x8b, 0x20, 0x39);
	addToMapW(0x8c, 0x00, 0x8C);
	addToMapW(0x8c, 0x01, 0x52);
	addToMapW(0x8c, 0x01, 0x53);
	addToMapW(0x8c, 0x03, 0x9D);
	addToMapW(0x8c, 0x03, 0xBD);
	addToMapW(0x8c, 0x04, 0x1C);
	addToMapW(0x8c, 0x04, 0x3C);
	addToMapW(0x8c, 0x05, 0xDC);
	addToMapW(0x8d, 0x00, 0x8D);
	addToMapW(0x8d, 0x01, 0x31);
	addToMapW(0x8d, 0x01, 0x79);
	addToMapW(0x8d, 0x01, 0x7A);
	addToMapW(0x8d, 0x03, 0x9E);
	addToMapW(0x8d, 0x03, 0xBE);
	addToMapW(0x8d, 0x04, 0x07);
	addToMapW(0x8d, 0x04, 0x1D);
	addToMapW(0x8d, 0x04, 0x3D);
	addToMapW(0x8d, 0x04, 0x57);
	addToMapW(0x8d, 0x05, 0xDD);
	addToMapW(0x8e, 0x00, 0x8E);
	addToMapW(0x8e, 0x00, 0xC4);
	addToMapW(0x8e, 0x00, 0xE4);
	addToMapW(0x8e, 0x03, 0x9F);
	addToMapW(0x8e, 0x03, 0xBF);
	addToMapW(0x8e, 0x04, 0x1E);
	addToMapW(0x8e, 0x04, 0x3E);
	addToMapW(0x8e, 0x05, 0xDE);
	addToMapW(0x8f, 0x00, 0x8F);
	addToMapW(0x8f, 0x00, 0xC5);
	addToMapW(0x8f, 0x00, 0xE5);
	addToMapW(0x8f, 0x01, 0x06);
	addToMapW(0x8f, 0x01, 0x07);
	addToMapW(0x8f, 0x03, 0xA0);
	addToMapW(0x8f, 0x03, 0xC0);
	addToMapW(0x8f, 0x04, 0x08);
	addToMapW(0x8f, 0x04, 0x1F);
	addToMapW(0x8f, 0x04, 0x3F);
	addToMapW(0x8f, 0x04, 0x58);
	addToMapW(0x8f, 0x05, 0xDF);
	addToMapW(0x8f, 0x21, 0x2B);
	addToMapW(0x90, 0x00, 0x90);
	addToMapW(0x90, 0x00, 0xC9);
	addToMapW(0x90, 0x00, 0xE9);
	addToMapW(0x90, 0x03, 0xA1);
	addToMapW(0x90, 0x03, 0xC1);
	addToMapW(0x90, 0x04, 0x20);
	addToMapW(0x90, 0x04, 0x40);
	addToMapW(0x90, 0x05, 0xE0);
	addToMapW(0x91, 0x01, 0x39);
	addToMapW(0x91, 0x01, 0x3A);
	addToMapW(0x91, 0x03, 0xA3);
	addToMapW(0x91, 0x03, 0xC2);
	addToMapW(0x91, 0x03, 0xC3);
	addToMapW(0x91, 0x04, 0x09);
	addToMapW(0x91, 0x04, 0x21);
	addToMapW(0x91, 0x04, 0x41);
	addToMapW(0x91, 0x04, 0x59);
	addToMapW(0x91, 0x05, 0xE1);
	addToMapW(0x91, 0x06, 0x51);
	addToMapW(0x91, 0x20, 0x18);
	addToMapW(0x91, 0xFE, 0x7C);
	addToMapW(0x91, 0xFE, 0x7D);
	addToMapW(0x92, 0x00, 0xC6);
	addToMapW(0x92, 0x00, 0xE6);
	addToMapW(0x92, 0x03, 0xA4);
	addToMapW(0x92, 0x03, 0xC4);
	addToMapW(0x92, 0x04, 0x22);
	addToMapW(0x92, 0x04, 0x42);
	addToMapW(0x92, 0x05, 0xE2);
	addToMapW(0x92, 0x06, 0x52);
	addToMapW(0x92, 0x20, 0x19);
	addToMapW(0x92, 0xFE, 0x7E);
	addToMapW(0x92, 0xFE, 0x7F);
	addToMapW(0x93, 0x03, 0xA5);
	addToMapW(0x93, 0x03, 0xC5);
	addToMapW(0x93, 0x04, 0x0A);
	addToMapW(0x93, 0x04, 0x23);
	addToMapW(0x93, 0x04, 0x43);
	addToMapW(0x93, 0x04, 0x5A);
	addToMapW(0x93, 0x05, 0xE3);
	addToMapW(0x93, 0x20, 0x1C);
	addToMapW(0x94, 0x00, 0xA4);
	addToMapW(0x94, 0x03, 0xA6);
	addToMapW(0x94, 0x03, 0xC6);
	addToMapW(0x94, 0x04, 0x24);
	addToMapW(0x94, 0x04, 0x44);
	addToMapW(0x94, 0x05, 0xE4);
	addToMapW(0x94, 0x20, 0x1D);
	addToMapW(0x95, 0x01, 0x22);
	addToMapW(0x95, 0x01, 0x23);
	addToMapW(0x95, 0x01, 0x3D);
	addToMapW(0x95, 0x01, 0x3E);
	addToMapW(0x95, 0x03, 0xA7);
	addToMapW(0x95, 0x03, 0xC7);
	addToMapW(0x95, 0x04, 0x0B);
	addToMapW(0x95, 0x04, 0x25);
	addToMapW(0x95, 0x04, 0x45);
	addToMapW(0x95, 0x04, 0x5B);
	addToMapW(0x95, 0x05, 0xE5);
	addToMapW(0x95, 0x06, 0x40);
	addToMapW(0x95, 0x20, 0x22);
	addToMapW(0x96, 0x00, 0xA2);
	addToMapW(0x96, 0x03, 0xA8);
	addToMapW(0x96, 0x03, 0xC8);
	addToMapW(0x96, 0x04, 0x26);
	addToMapW(0x96, 0x04, 0x46);
	addToMapW(0x96, 0x05, 0xE6);
	addToMapW(0x96, 0x20, 0x13);
	addToMapW(0x97, 0x00, 0xB5);
	addToMapW(0x97, 0x01, 0x5A);
	addToMapW(0x97, 0x01, 0x5B);
	addToMapW(0x97, 0x03, 0xA9);
	addToMapW(0x97, 0x03, 0xC9);
	addToMapW(0x97, 0x04, 0x0C);
	addToMapW(0x97, 0x04, 0x27);
	addToMapW(0x97, 0x04, 0x47);
	addToMapW(0x97, 0x04, 0x5C);
	addToMapW(0x97, 0x05, 0xE7);
	addToMapW(0x97, 0x20, 0x14);
	addToMapW(0x98, 0x00, 0x98);
	addToMapW(0x98, 0x01, 0x30);
	addToMapW(0x98, 0x02, 0xDC);
	addToMapW(0x98, 0x04, 0x28);
	addToMapW(0x98, 0x04, 0x48);
	addToMapW(0x98, 0x05, 0xE8);
	addToMapW(0x98, 0x06, 0x21);
	addToMapW(0x98, 0xFE, 0x80);
	addToMapW(0x99, 0x00, 0x99);
	addToMapW(0x99, 0x00, 0xD6);
	addToMapW(0x99, 0x00, 0xF6);
	addToMapW(0x99, 0x04, 0x0E);
	addToMapW(0x99, 0x04, 0x29);
	addToMapW(0x99, 0x04, 0x49);
	addToMapW(0x99, 0x04, 0x5E);
	addToMapW(0x99, 0x05, 0xE9);
	addToMapW(0x99, 0x06, 0x22);
	addToMapW(0x99, 0x21, 0x22);
	addToMapW(0x99, 0xFE, 0x81);
	addToMapW(0x99, 0xFE, 0x82);
	addToMapW(0x9a, 0x00, 0x9A);
	addToMapW(0x9a, 0x00, 0xDC);
	addToMapW(0x9a, 0x00, 0xFC);
	addToMapW(0x9a, 0x04, 0x2A);
	addToMapW(0x9a, 0x04, 0x4A);
	addToMapW(0x9a, 0x05, 0xEA);
	addToMapW(0x9a, 0x06, 0x23);
	addToMapW(0x9a, 0xFE, 0x83);
	addToMapW(0x9a, 0xFE, 0x84);
	addToMapW(0x9b, 0x00, 0x9B);
	addToMapW(0x9b, 0x00, 0xA2);
	addToMapW(0x9b, 0x01, 0x64);
	addToMapW(0x9b, 0x01, 0x65);
	addToMapW(0x9b, 0x04, 0x0F);
	addToMapW(0x9b, 0x04, 0x2B);
	addToMapW(0x9b, 0x04, 0x4B);
	addToMapW(0x9b, 0x04, 0x5F);
	addToMapW(0x9b, 0x06, 0x24);
	addToMapW(0x9b, 0x20, 0x3A);
	addToMapW(0x9b, 0xFE, 0x85);
	addToMapW(0x9b, 0xFE, 0x86);
	addToMapW(0x9c, 0x00, 0x9C);
	addToMapW(0x9c, 0x00, 0xA3);
	addToMapW(0x9c, 0x04, 0x2C);
	addToMapW(0x9c, 0x04, 0x4C);
	addToMapW(0x9c, 0x20, 0xA4);
	addToMapW(0x9d, 0x00, 0x9D);
	addToMapW(0x9d, 0x00, 0xA5);
	addToMapW(0x9d, 0x00, 0xD8);
	addToMapW(0x9d, 0x00, 0xF8);
	addToMapW(0x9d, 0x01, 0x41);
	addToMapW(0x9d, 0x01, 0x42);
	addToMapW(0x9d, 0x02, 0x78);
	addToMapW(0x9d, 0x03, 0x98);
	addToMapW(0x9d, 0x04, 0x2D);
	addToMapW(0x9d, 0x04, 0x2E);
	addToMapW(0x9d, 0x04, 0x4D);
	addToMapW(0x9d, 0x04, 0x4E);
	addToMapW(0x9d, 0x06, 0x25);
	addToMapW(0x9d, 0x22, 0x05);
	addToMapW(0x9d, 0xFE, 0x87);
	addToMapW(0x9d, 0xFE, 0x88);
	addToMapW(0x9e, 0x00, 0x9E);
	addToMapW(0x9e, 0x00, 0xD7);
	addToMapW(0x9e, 0x01, 0x5E);
	addToMapW(0x9e, 0x01, 0x5F);
	addToMapW(0x9e, 0x04, 0x2E);
	addToMapW(0x9e, 0x04, 0x4E);
	addToMapW(0x9e, 0x06, 0x26);
	addToMapW(0x9e, 0x20, 0xA7);
	addToMapW(0x9e, 0xFE, 0x89);
	addToMapW(0x9e, 0xFE, 0x8A);
	addToMapW(0x9e, 0xFE, 0x8B);
	addToMapW(0x9e, 0xFE, 0x8C);
	addToMapW(0x9f, 0x00, 0x9F);
	addToMapW(0x9f, 0x00, 0xA4);
	addToMapW(0x9f, 0x00, 0xFF);
	addToMapW(0x9f, 0x01, 0x78);
	addToMapW(0x9f, 0x01, 0x91);
	addToMapW(0x9f, 0x01, 0x92);
	addToMapW(0x9f, 0x04, 0x2A);
	addToMapW(0x9f, 0x04, 0x2F);
	addToMapW(0x9f, 0x04, 0x4A);
	addToMapW(0x9f, 0x04, 0x4F);
	addToMapW(0x9f, 0x06, 0x27);
	addToMapW(0x9f, 0xFE, 0x8D);
	addToMapW(0x9f, 0xFE, 0x8E);
	addToMapW(0xa0, 0x00, 0xA0);
	addToMapW(0xa0, 0x01, 0x00);
	addToMapW(0xa0, 0x01, 0x01);
	addToMapW(0xa0, 0x06, 0x28);
	addToMapW(0xa0, 0xF8, 0xF0);
	addToMapW(0xa0, 0xFE, 0x8F);
	addToMapW(0xa0, 0xFE, 0x90);
	addToMapW(0xa0, 0xFE, 0x91);
	addToMapW(0xa0, 0xFE, 0x92);
	addToMapW(0xa1, 0x00, 0xA1);
	addToMapW(0xa1, 0x01, 0x2A);
	addToMapW(0xa1, 0x01, 0x2B);
	addToMapW(0xa1, 0x04, 0x10);
	addToMapW(0xa1, 0x04, 0x30);
	addToMapW(0xa1, 0x06, 0x29);
	addToMapW(0xa1, 0x0E, 0x01);
	addToMapW(0xa1, 0xFE, 0x93);
	addToMapW(0xa1, 0xFE, 0x94);
	addToMapW(0xa1, 0xFF, 0x61);
	addToMapW(0xa2, 0x00, 0xA2);
	addToMapW(0xa2, 0x06, 0x2A);
	addToMapW(0xa2, 0x0E, 0x02);
	addToMapW(0xa2, 0xFE, 0x95);
	addToMapW(0xa2, 0xFE, 0x96);
	addToMapW(0xa2, 0xFE, 0x97);
	addToMapW(0xa2, 0xFE, 0x98);
	addToMapW(0xa2, 0xFF, 0x62);
	addToMapW(0xa3, 0x00, 0xA3);
	addToMapW(0xa3, 0x01, 0x7B);
	addToMapW(0xa3, 0x01, 0x7C);
	addToMapW(0xa3, 0x04, 0x11);
	addToMapW(0xa3, 0x04, 0x31);
	addToMapW(0xa3, 0x06, 0x2B);
	addToMapW(0xa3, 0x0E, 0x03);
	addToMapW(0xa3, 0xFE, 0x99);
	addToMapW(0xa3, 0xFE, 0x9A);
	addToMapW(0xa3, 0xFE, 0x9B);
	addToMapW(0xa3, 0xFE, 0x9C);
	addToMapW(0xa3, 0xFF, 0x63);
	addToMapW(0xa4, 0x00, 0xA4);
	addToMapW(0xa4, 0x01, 0x04);
	addToMapW(0xa4, 0x01, 0x05);
	addToMapW(0xa4, 0x06, 0x2C);
	addToMapW(0xa4, 0x0E, 0x04);
	addToMapW(0xa4, 0xFE, 0x9D);
	addToMapW(0xa4, 0xFE, 0x9E);
	addToMapW(0xa4, 0xFE, 0x9F);
	addToMapW(0xa4, 0xFE, 0xA0);
	addToMapW(0xa4, 0xFF, 0x64);
	addToMapW(0xa5, 0x00, 0xA5);
	addToMapW(0xa5, 0x00, 0xD1);
	addToMapW(0xa5, 0x00, 0xF1);
	addToMapW(0xa5, 0x04, 0x26);
	addToMapW(0xa5, 0x04, 0x46);
	addToMapW(0xa5, 0x06, 0x2D);
	addToMapW(0xa5, 0x0E, 0x05);
	addToMapW(0xa5, 0xFE, 0xA1);
	addToMapW(0xa5, 0xFE, 0xA2);
	addToMapW(0xa5, 0xFE, 0xA3);
	addToMapW(0xa5, 0xFE, 0xA4);
	addToMapW(0xa5, 0xFF, 0x65);
	addToMapW(0xa6, 0x00, 0xA6);
	addToMapW(0xa6, 0x00, 0xAA);
	addToMapW(0xa6, 0x01, 0x1E);
	addToMapW(0xa6, 0x01, 0x1F);
	addToMapW(0xa6, 0x01, 0x7D);
	addToMapW(0xa6, 0x01, 0x7E);
	addToMapW(0xa6, 0x06, 0x2E);
	addToMapW(0xa6, 0x0E, 0x06);
	addToMapW(0xa6, 0x20, 0x1D);
	addToMapW(0xa6, 0xFE, 0xA5);
	addToMapW(0xa6, 0xFE, 0xA6);
	addToMapW(0xa6, 0xFE, 0xA7);
	addToMapW(0xa6, 0xFE, 0xA8);
	addToMapW(0xa6, 0xFF, 0x66);
	addToMapW(0xa7, 0x00, 0xA6);
	addToMapW(0xa7, 0x00, 0xA7);
	addToMapW(0xa7, 0x00, 0xBA);
	addToMapW(0xa7, 0x04, 0x14);
	addToMapW(0xa7, 0x04, 0x34);
	addToMapW(0xa7, 0x06, 0x2F);
	addToMapW(0xa7, 0x0E, 0x07);
	addToMapW(0xa7, 0xFE, 0xA9);
	addToMapW(0xa7, 0xFE, 0xAA);
	addToMapW(0xa7, 0xFF, 0x67);
	addToMapW(0xa8, 0x00, 0xA8);
	addToMapW(0xa8, 0x00, 0xA9);
	addToMapW(0xa8, 0x00, 0xBF);
	addToMapW(0xa8, 0x01, 0x18);
	addToMapW(0xa8, 0x01, 0x19);
	addToMapW(0xa8, 0x06, 0x30);
	addToMapW(0xa8, 0x0E, 0x08);
	addToMapW(0xa8, 0xFE, 0xAB);
	addToMapW(0xa8, 0xFE, 0xAC);
	addToMapW(0xa8, 0xFF, 0x68);
	addToMapW(0xa9, 0x00, 0xA9);
	addToMapW(0xa9, 0x00, 0xAE);
	addToMapW(0xa9, 0x04, 0x15);
	addToMapW(0xa9, 0x04, 0x35);
	addToMapW(0xa9, 0x06, 0x31);
	addToMapW(0xa9, 0x0E, 0x09);
	addToMapW(0xa9, 0x23, 0x10);
	addToMapW(0xa9, 0xFE, 0xAD);
	addToMapW(0xa9, 0xFE, 0xAE);
	addToMapW(0xa9, 0xFF, 0x69);
	addToMapW(0xaa, 0x00, 0xAA);
	addToMapW(0xaa, 0x00, 0xAC);
	addToMapW(0xaa, 0x06, 0x32);
	addToMapW(0xaa, 0x0E, 0x0A);
	addToMapW(0xaa, 0x23, 0x10);
	addToMapW(0xaa, 0xFE, 0xAF);
	addToMapW(0xaa, 0xFE, 0xB0);
	addToMapW(0xaa, 0xFF, 0x6A);
	addToMapW(0xab, 0x00, 0xAB);
	addToMapW(0xab, 0x00, 0xBD);
	addToMapW(0xab, 0x04, 0x24);
	addToMapW(0xab, 0x04, 0x44);
	addToMapW(0xab, 0x06, 0x33);
	addToMapW(0xab, 0x0E, 0x0B);
	addToMapW(0xab, 0xFE, 0xB1);
	addToMapW(0xab, 0xFE, 0xB2);
	addToMapW(0xab, 0xFE, 0xB3);
	addToMapW(0xab, 0xFE, 0xB4);
	addToMapW(0xab, 0xFF, 0x6B);
	addToMapW(0xac, 0x00, 0xAC);
	addToMapW(0xac, 0x00, 0xBC);
	addToMapW(0xac, 0x01, 0x0C);
	addToMapW(0xac, 0x01, 0x0D);
	addToMapW(0xac, 0x06, 0x34);
	addToMapW(0xac, 0x0E, 0x0C);
	addToMapW(0xac, 0xFE, 0xB5);
	addToMapW(0xac, 0xFE, 0xB6);
	addToMapW(0xac, 0xFE, 0xB7);
	addToMapW(0xac, 0xFE, 0xB8);
	addToMapW(0xac, 0xFF, 0x6C);
	addToMapW(0xad, 0x00, 0xA1);
	addToMapW(0xad, 0x00, 0xAD);
	addToMapW(0xad, 0x01, 0x41);
	addToMapW(0xad, 0x01, 0x42);
	addToMapW(0xad, 0x04, 0x13);
	addToMapW(0xad, 0x04, 0x33);
	addToMapW(0xad, 0x06, 0x35);
	addToMapW(0xad, 0x0E, 0x0D);
	addToMapW(0xad, 0xFE, 0xB9);
	addToMapW(0xad, 0xFE, 0xBA);
	addToMapW(0xad, 0xFE, 0xBB);
	addToMapW(0xad, 0xFE, 0xBC);
	addToMapW(0xad, 0xFF, 0x6D);
	addToMapW(0xae, 0x00, 0xAB);
	addToMapW(0xae, 0x00, 0xAE);
	addToMapW(0xae, 0x0E, 0x0E);
	addToMapW(0xae, 0x22, 0x6A);
	addToMapW(0xae, 0x30, 0x0A);
	addToMapW(0xae, 0xFF, 0x6E);
	addToMapW(0xaf, 0x00, 0xAF);
	addToMapW(0xaf, 0x00, 0xBB);
	addToMapW(0xaf, 0x0E, 0x0F);
	addToMapW(0xaf, 0x22, 0x6B);
	addToMapW(0xaf, 0x30, 0x0B);
	addToMapW(0xaf, 0xFF, 0x6F);
	addToMapW(0xb0, 0x00, 0xB0);
	addToMapW(0xb0, 0x0E, 0x10);
	addToMapW(0xb0, 0x25, 0x91);
	addToMapW(0xb0, 0xFF, 0x70);
	addToMapW(0xb1, 0x00, 0xB1);
	addToMapW(0xb1, 0x0E, 0x11);
	addToMapW(0xb1, 0x25, 0x92);
	addToMapW(0xb1, 0xFF, 0x71);
	addToMapW(0xb2, 0x00, 0xB2);
	addToMapW(0xb2, 0x0E, 0x12);
	addToMapW(0xb2, 0x25, 0x93);
	addToMapW(0xb2, 0xFF, 0x72);
	addToMapW(0xb3, 0x00, 0xA6);
	addToMapW(0xb3, 0x00, 0xB3);
	addToMapW(0xb3, 0x01, 0xC0);
	addToMapW(0xb3, 0x0E, 0x13);
	addToMapW(0xb3, 0x22, 0x23);
	addToMapW(0xb3, 0x25, 0x02);
	addToMapW(0xb3, 0x27, 0x58);
	addToMapW(0xb3, 0xFF, 0x73);
	addToMapW(0xb4, 0x00, 0xB4);
	addToMapW(0xb4, 0x0E, 0x14);
	addToMapW(0xb4, 0x25, 0x24);
	addToMapW(0xb4, 0xFF, 0x74);
	addToMapW(0xb5, 0x00, 0xB5);
	addToMapW(0xb5, 0x00, 0xC1);
	addToMapW(0xb5, 0x00, 0xE1);
	addToMapW(0xb5, 0x01, 0x04);
	addToMapW(0xb5, 0x01, 0x05);
	addToMapW(0xb5, 0x0E, 0x15);
	addToMapW(0xb5, 0x25, 0x61);
	addToMapW(0xb5, 0xFF, 0x75);
	addToMapW(0xb6, 0x00, 0xB6);
	addToMapW(0xb6, 0x00, 0xC2);
	addToMapW(0xb6, 0x00, 0xE2);
	addToMapW(0xb6, 0x01, 0x0C);
	addToMapW(0xb6, 0x01, 0x0D);
	addToMapW(0xb6, 0x04, 0x25);
	addToMapW(0xb6, 0x04, 0x45);
	addToMapW(0xb6, 0x0E, 0x16);
	addToMapW(0xb6, 0x25, 0x62);
	addToMapW(0xb6, 0xFF, 0x76);
	addToMapW(0xb7, 0x00, 0xB7);
	addToMapW(0xb7, 0x00, 0xC0);
	addToMapW(0xb7, 0x00, 0xE0);
	addToMapW(0xb7, 0x01, 0x18);
	addToMapW(0xb7, 0x01, 0x19);
	addToMapW(0xb7, 0x01, 0x1A);
	addToMapW(0xb7, 0x01, 0x1B);
	addToMapW(0xb7, 0x0E, 0x17);
	addToMapW(0xb7, 0x25, 0x56);
	addToMapW(0xb7, 0xFF, 0x77);
	addToMapW(0xb8, 0x00, 0xA9);
	addToMapW(0xb8, 0x00, 0xB8);
	addToMapW(0xb8, 0x01, 0x16);
	addToMapW(0xb8, 0x01, 0x17);
	addToMapW(0xb8, 0x01, 0x5E);
	addToMapW(0xb8, 0x01, 0x5F);
	addToMapW(0xb8, 0x04, 0x18);
	addToMapW(0xb8, 0x04, 0x38);
	addToMapW(0xb8, 0x0E, 0x18);
	addToMapW(0xb8, 0x25, 0x55);
	addToMapW(0xb8, 0xFF, 0x78);
	addToMapW(0xb9, 0x00, 0xB9);
	addToMapW(0xb9, 0x0E, 0x19);
	addToMapW(0xb9, 0x25, 0x61);
	addToMapW(0xb9, 0x25, 0x62);
	addToMapW(0xb9, 0x25, 0x63);
	addToMapW(0xb9, 0xFF, 0x79);
	addToMapW(0xba, 0x00, 0xBA);
	addToMapW(0xba, 0x0E, 0x1A);
	addToMapW(0xba, 0x25, 0x51);
	addToMapW(0xba, 0xFF, 0x7A);
	addToMapW(0xbb, 0x00, 0xBB);
	addToMapW(0xbb, 0x0E, 0x1B);
	addToMapW(0xbb, 0x25, 0x55);
	addToMapW(0xbb, 0x25, 0x56);
	addToMapW(0xbb, 0x25, 0x57);
	addToMapW(0xbb, 0xFF, 0x7B);
	addToMapW(0xbc, 0x00, 0xBC);
	addToMapW(0xbc, 0x0E, 0x1C);
	addToMapW(0xbc, 0x25, 0x5B);
	addToMapW(0xbc, 0x25, 0x5C);
	addToMapW(0xbc, 0x25, 0x5D);
	addToMapW(0xbc, 0xFF, 0x7C);
	addToMapW(0xbd, 0x00, 0xA2);
	addToMapW(0xbd, 0x00, 0xBD);
	addToMapW(0xbd, 0x01, 0x2E);
	addToMapW(0xbd, 0x01, 0x2F);
	addToMapW(0xbd, 0x01, 0x7B);
	addToMapW(0xbd, 0x01, 0x7C);
	addToMapW(0xbd, 0x0E, 0x1D);
	addToMapW(0xbd, 0x25, 0x5C);
	addToMapW(0xbd, 0xFF, 0x7D);
	addToMapW(0xbe, 0x00, 0xA5);
	addToMapW(0xbe, 0x00, 0xBE);
	addToMapW(0xbe, 0x01, 0x60);
	addToMapW(0xbe, 0x01, 0x61);
	addToMapW(0xbe, 0x04, 0x19);
	addToMapW(0xbe, 0x04, 0x39);
	addToMapW(0xbe, 0x0E, 0x1E);
	addToMapW(0xbe, 0x25, 0x5B);
	addToMapW(0xbe, 0xFF, 0x7E);
	addToMapW(0xbf, 0x00, 0xAC);
	addToMapW(0xbf, 0x00, 0xBF);
	addToMapW(0xbf, 0x0E, 0x1F);
	addToMapW(0xbf, 0x25, 0x10);
	addToMapW(0xbf, 0xFF, 0x7F);
	addToMapW(0xc0, 0x00, 0xC0);
	addToMapW(0xc0, 0x00, 0xE0);
	addToMapW(0xc0, 0x0E, 0x20);
	addToMapW(0xc0, 0x25, 0x14);
	addToMapW(0xc0, 0xFF, 0x80);
	addToMapW(0xc1, 0x00, 0xC1);
	addToMapW(0xc1, 0x00, 0xE1);
	addToMapW(0xc1, 0x0E, 0x21);
	addToMapW(0xc1, 0x25, 0x34);
	addToMapW(0xc1, 0xFF, 0x81);
	addToMapW(0xc2, 0x00, 0xC2);
	addToMapW(0xc2, 0x00, 0xE2);
	addToMapW(0xc2, 0x0E, 0x22);
	addToMapW(0xc2, 0x25, 0x2C);
	addToMapW(0xc2, 0xFF, 0x82);
	addToMapW(0xc3, 0x01, 0x02);
	addToMapW(0xc3, 0x01, 0x03);
	addToMapW(0xc3, 0x0E, 0x23);
	addToMapW(0xc3, 0x25, 0x1C);
	addToMapW(0xc3, 0xFF, 0x83);
	addToMapW(0xc4, 0x00, 0xAF);
	addToMapW(0xc4, 0x00, 0xC4);
	addToMapW(0xc4, 0x00, 0xE4);
	addToMapW(0xc4, 0x02, 0xC9);
	addToMapW(0xc4, 0x03, 0x04);
	addToMapW(0xc4, 0x03, 0x05);
	addToMapW(0xc4, 0x0E, 0x24);
	addToMapW(0xc4, 0x25, 0x00);
	addToMapW(0xc4, 0xFF, 0x84);
	addToMapW(0xc5, 0x00, 0xC5);
	addToMapW(0xc5, 0x00, 0xE5);
	addToMapW(0xc5, 0x0E, 0x25);
	addToMapW(0xc5, 0x20, 0x20);
	addToMapW(0xc5, 0x20, 0x21);
	addToMapW(0xc5, 0x25, 0x3C);
	addToMapW(0xc5, 0xFF, 0x85);
	addToMapW(0xc6, 0x00, 0xC6);
	addToMapW(0xc6, 0x00, 0xE6);
	addToMapW(0xc6, 0x01, 0x02);
	addToMapW(0xc6, 0x01, 0x03);
	addToMapW(0xc6, 0x01, 0x72);
	addToMapW(0xc6, 0x01, 0x73);
	addToMapW(0xc6, 0x0E, 0x26);
	addToMapW(0xc6, 0x25, 0x5E);
	addToMapW(0xc6, 0xFF, 0x86);
	addToMapW(0xc7, 0x00, 0xC3);
	addToMapW(0xc7, 0x00, 0xC7);
	addToMapW(0xc7, 0x00, 0xE3);
	addToMapW(0xc7, 0x00, 0xE7);
	addToMapW(0xc7, 0x01, 0x6A);
	addToMapW(0xc7, 0x01, 0x6B);
	addToMapW(0xc7, 0x04, 0x1A);
	addToMapW(0xc7, 0x04, 0x3A);
	addToMapW(0xc7, 0x0E, 0x27);
	addToMapW(0xc7, 0x25, 0x5F);
	addToMapW(0xc7, 0xFF, 0x87);
	addToMapW(0xc8, 0x00, 0xC8);
	addToMapW(0xc8, 0x00, 0xE8);
	addToMapW(0xc8, 0x0E, 0x28);
	addToMapW(0xc8, 0x25, 0x58);
	addToMapW(0xc8, 0x25, 0x59);
	addToMapW(0xc8, 0x25, 0x5A);
	addToMapW(0xc8, 0xFF, 0x88);
	addToMapW(0xc9, 0x00, 0xC9);
	addToMapW(0xc9, 0x00, 0xE9);
	addToMapW(0xc9, 0x0E, 0x29);
	addToMapW(0xc9, 0x25, 0x52);
	addToMapW(0xc9, 0x25, 0x53);
	addToMapW(0xc9, 0x25, 0x54);
	addToMapW(0xc9, 0xFF, 0x89);
	addToMapW(0xca, 0x00, 0xCA);
	addToMapW(0xca, 0x00, 0xEA);
	addToMapW(0xca, 0x0E, 0x2A);
	addToMapW(0xca, 0x25, 0x67);
	addToMapW(0xca, 0x25, 0x68);
	addToMapW(0xca, 0x25, 0x69);
	addToMapW(0xca, 0xFF, 0x8A);
	addToMapW(0xcb, 0x00, 0xCB);
	addToMapW(0xcb, 0x00, 0xEB);
	addToMapW(0xcb, 0x0E, 0x2B);
	addToMapW(0xcb, 0x25, 0x64);
	addToMapW(0xcb, 0x25, 0x65);
	addToMapW(0xcb, 0x25, 0x66);
	addToMapW(0xcb, 0xFF, 0x8B);
	addToMapW(0xcc, 0x03, 0x00);
	addToMapW(0xcc, 0x0E, 0x2C);
	addToMapW(0xcc, 0x25, 0x5E);
	addToMapW(0xcc, 0x25, 0x5F);
	addToMapW(0xcc, 0x25, 0x60);
	addToMapW(0xcc, 0xFF, 0x8C);
	addToMapW(0xcd, 0x00, 0xCD);
	addToMapW(0xcd, 0x00, 0xED);
	addToMapW(0xcd, 0x0E, 0x2D);
	addToMapW(0xcd, 0x25, 0x50);
	addToMapW(0xcd, 0xFF, 0x8D);
	addToMapW(0xce, 0x00, 0xCE);
	addToMapW(0xce, 0x00, 0xEE);
	addToMapW(0xce, 0x0E, 0x2E);
	addToMapW(0xce, 0x20, 0x21);
	addToMapW(0xce, 0x25, 0x6A);
	addToMapW(0xce, 0x25, 0x6B);
	addToMapW(0xce, 0x25, 0x6C);
	addToMapW(0xce, 0xFF, 0x8E);
	addToMapW(0xcf, 0x00, 0xA4);
	addToMapW(0xcf, 0x00, 0xCF);
	addToMapW(0xcf, 0x00, 0xEF);
	addToMapW(0xcf, 0x01, 0x7D);
	addToMapW(0xcf, 0x01, 0x7E);
	addToMapW(0xcf, 0x0E, 0x2F);
	addToMapW(0xcf, 0x25, 0x67);
	addToMapW(0xcf, 0xFF, 0x8F);
	addToMapW(0xd0, 0x00, 0xBA);
	addToMapW(0xd0, 0x01, 0x10);
	addToMapW(0xd0, 0x01, 0x11);
	addToMapW(0xd0, 0x0E, 0x30);
	addToMapW(0xd0, 0x25, 0x68);
	addToMapW(0xd0, 0xFF, 0x90);
	addToMapW(0xd1, 0x00, 0xAA);
	addToMapW(0xd1, 0x00, 0xD0);
	addToMapW(0xd1, 0x00, 0xD1);
	addToMapW(0xd1, 0x00, 0xF0);
	addToMapW(0xd1, 0x00, 0xF1);
	addToMapW(0xd1, 0x01, 0x10);
	addToMapW(0xd1, 0x01, 0x11);
	addToMapW(0xd1, 0x01, 0x89);
	addToMapW(0xd1, 0x04, 0x1B);
	addToMapW(0xd1, 0x04, 0x3B);
	addToMapW(0xd1, 0x0E, 0x31);
	addToMapW(0xd1, 0x25, 0x64);
	addToMapW(0xd1, 0xFF, 0x91);
	addToMapW(0xd2, 0x00, 0xCA);
	addToMapW(0xd2, 0x00, 0xEA);
	addToMapW(0xd2, 0x01, 0x0E);
	addToMapW(0xd2, 0x01, 0x0F);
	addToMapW(0xd2, 0x03, 0x09);
	addToMapW(0xd2, 0x0E, 0x32);
	addToMapW(0xd2, 0x25, 0x65);
	addToMapW(0xd2, 0xFF, 0x92);
	addToMapW(0xd3, 0x00, 0xCB);
	addToMapW(0xd3, 0x00, 0xD3);
	addToMapW(0xd3, 0x00, 0xEB);
	addToMapW(0xd3, 0x00, 0xF3);
	addToMapW(0xd3, 0x04, 0x1C);
	addToMapW(0xd3, 0x04, 0x3C);
	addToMapW(0xd3, 0x0E, 0x33);
	addToMapW(0xd3, 0x25, 0x59);
	addToMapW(0xd3, 0xFF, 0x93);
	addToMapW(0xd4, 0x00, 0xC8);
	addToMapW(0xd4, 0x00, 0xD4);
	addToMapW(0xd4, 0x00, 0xE8);
	addToMapW(0xd4, 0x00, 0xF4);
	addToMapW(0xd4, 0x0E, 0x34);
	addToMapW(0xd4, 0x25, 0x58);
	addToMapW(0xd4, 0xFF, 0x94);
	addToMapW(0xd5, 0x01, 0x31);
	addToMapW(0xd5, 0x01, 0x47);
	addToMapW(0xd5, 0x01, 0x48);
	addToMapW(0xd5, 0x01, 0xA0);
	addToMapW(0xd5, 0x01, 0xA1);
	addToMapW(0xd5, 0x04, 0x1D);
	addToMapW(0xd5, 0x04, 0x3D);
	addToMapW(0xd5, 0x0E, 0x35);
	addToMapW(0xd5, 0x25, 0x52);
	addToMapW(0xd5, 0xF8, 0xBB);
	addToMapW(0xd5, 0xFF, 0x95);
	addToMapW(0xd6, 0x00, 0xCD);
	addToMapW(0xd6, 0x00, 0xD6);
	addToMapW(0xd6, 0x00, 0xED);
	addToMapW(0xd6, 0x00, 0xF6);
	addToMapW(0xd6, 0x0E, 0x36);
	addToMapW(0xd6, 0x25, 0x53);
	addToMapW(0xd6, 0xFF, 0x96);
	addToMapW(0xd7, 0x00, 0xCE);
	addToMapW(0xd7, 0x00, 0xD7);
	addToMapW(0xd7, 0x00, 0xEE);
	addToMapW(0xd7, 0x04, 0x1E);
	addToMapW(0xd7, 0x04, 0x3E);
	addToMapW(0xd7, 0x0E, 0x37);
	addToMapW(0xd7, 0x25, 0x6B);
	addToMapW(0xd7, 0xFF, 0x97);
	addToMapW(0xd8, 0x00, 0xCF);
	addToMapW(0xd8, 0x00, 0xD8);
	addToMapW(0xd8, 0x00, 0xEF);
	addToMapW(0xd8, 0x00, 0xF8);
	addToMapW(0xd8, 0x0E, 0x38);
	addToMapW(0xd8, 0x20, 0x21);
	addToMapW(0xd8, 0x25, 0x6A);
	addToMapW(0xd8, 0xFF, 0x98);
	addToMapW(0xd9, 0x00, 0xD9);
	addToMapW(0xd9, 0x00, 0xF9);
	addToMapW(0xd9, 0x0E, 0x39);
	addToMapW(0xd9, 0x25, 0x18);
	addToMapW(0xd9, 0xFF, 0x99);
	addToMapW(0xda, 0x00, 0xDA);
	addToMapW(0xda, 0x00, 0xFA);
	addToMapW(0xda, 0x0E, 0x3A);
	addToMapW(0xda, 0x25, 0x0C);
	addToMapW(0xda, 0xFF, 0x9A);
	addToMapW(0xdb, 0x00, 0xDB);
	addToMapW(0xdb, 0x00, 0xFB);
	addToMapW(0xdb, 0x25, 0x88);
	addToMapW(0xdb, 0x25, 0x8C);
	addToMapW(0xdb, 0x25, 0x90);
	addToMapW(0xdb, 0xF8, 0xC1);
	addToMapW(0xdb, 0xFF, 0x9B);
	addToMapW(0xdc, 0x00, 0xDC);
	addToMapW(0xdc, 0x00, 0xFC);
	addToMapW(0xdc, 0x25, 0x84);
	addToMapW(0xdc, 0xF8, 0xC2);
	addToMapW(0xdc, 0xFF, 0x9C);
	addToMapW(0xdd, 0x00, 0xA6);
	addToMapW(0xdd, 0x01, 0x62);
	addToMapW(0xdd, 0x01, 0x63);
	addToMapW(0xdd, 0x01, 0xAF);
	addToMapW(0xdd, 0x01, 0xB0);
	addToMapW(0xdd, 0x04, 0x1F);
	addToMapW(0xdd, 0x04, 0x3F);
	addToMapW(0xdd, 0x25, 0x8C);
	addToMapW(0xdd, 0xF8, 0xC3);
	addToMapW(0xdd, 0xFF, 0x9D);
	addToMapW(0xde, 0x00, 0xCC);
	addToMapW(0xde, 0x00, 0xEC);
	addToMapW(0xde, 0x01, 0x6E);
	addToMapW(0xde, 0x01, 0x6F);
	addToMapW(0xde, 0x03, 0x03);
	addToMapW(0xde, 0x25, 0x90);
	addToMapW(0xde, 0xF8, 0xC4);
	addToMapW(0xde, 0xFF, 0x9E);
	addToMapW(0xdf, 0x00, 0xDF);
	addToMapW(0xdf, 0x0E, 0x3F);
	addToMapW(0xdf, 0x25, 0x80);
	addToMapW(0xdf, 0xFF, 0x9F);
	addToMapW(0xe0, 0x00, 0xD3);
	addToMapW(0xe0, 0x00, 0xF3);
	addToMapW(0xe0, 0x03, 0x91);
	addToMapW(0xe0, 0x03, 0xB1);
	addToMapW(0xe0, 0x04, 0x2F);
	addToMapW(0xe0, 0x04, 0x4F);
	addToMapW(0xe0, 0x06, 0x36);
	addToMapW(0xe0, 0x0E, 0x40);
	addToMapW(0xe0, 0xFE, 0xBD);
	addToMapW(0xe0, 0xFE, 0xBE);
	addToMapW(0xe0, 0xFE, 0xBF);
	addToMapW(0xe0, 0xFE, 0xC0);
	addToMapW(0xe1, 0x00, 0xDF);
	addToMapW(0xe1, 0x03, 0xB2);
	addToMapW(0xe1, 0x06, 0x37);
	addToMapW(0xe1, 0x0E, 0x41);
	addToMapW(0xe1, 0xFE, 0xC1);
	addToMapW(0xe1, 0xFE, 0xC2);
	addToMapW(0xe1, 0xFE, 0xC3);
	addToMapW(0xe1, 0xFE, 0xC4);
	addToMapW(0xe2, 0x00, 0xD4);
	addToMapW(0xe2, 0x00, 0xF4);
	addToMapW(0xe2, 0x01, 0x4C);
	addToMapW(0xe2, 0x01, 0x4D);
	addToMapW(0xe2, 0x03, 0x93);
	addToMapW(0xe2, 0x04, 0x20);
	addToMapW(0xe2, 0x04, 0x40);
	addToMapW(0xe2, 0x06, 0x38);
	addToMapW(0xe2, 0x0E, 0x42);
	addToMapW(0xe2, 0xFE, 0xC5);
	addToMapW(0xe2, 0xFE, 0xC6);
	addToMapW(0xe2, 0xFE, 0xC7);
	addToMapW(0xe2, 0xFE, 0xC8);
	addToMapW(0xe3, 0x00, 0xD2);
	addToMapW(0xe3, 0x00, 0xF2);
	addToMapW(0xe3, 0x01, 0x43);
	addToMapW(0xe3, 0x01, 0x44);
	addToMapW(0xe3, 0x03, 0xA0);
	addToMapW(0xe3, 0x03, 0xC0);
	addToMapW(0xe3, 0x06, 0x39);
	addToMapW(0xe3, 0x0E, 0x43);
	addToMapW(0xe3, 0xFE, 0xC9);
	addToMapW(0xe3, 0xFE, 0xCA);
	addToMapW(0xe3, 0xFE, 0xCB);
	addToMapW(0xe3, 0xFE, 0xCC);
	addToMapW(0xe4, 0x01, 0xA9);
	addToMapW(0xe4, 0x03, 0xA3);
	addToMapW(0xe4, 0x03, 0xC3);
	addToMapW(0xe4, 0x04, 0x21);
	addToMapW(0xe4, 0x04, 0x41);
	addToMapW(0xe4, 0x06, 0x3A);
	addToMapW(0xe4, 0x0E, 0x44);
	addToMapW(0xe4, 0x22, 0x11);
	addToMapW(0xe4, 0xFE, 0xCD);
	addToMapW(0xe4, 0xFE, 0xCE);
	addToMapW(0xe4, 0xFE, 0xCF);
	addToMapW(0xe4, 0xFE, 0xD0);
	addToMapW(0xe5, 0x00, 0xD5);
	addToMapW(0xe5, 0x00, 0xF5);
	addToMapW(0xe5, 0x06, 0x41);
	addToMapW(0xe5, 0x0E, 0x45);
	addToMapW(0xe5, 0xFE, 0xD1);
	addToMapW(0xe5, 0xFE, 0xD2);
	addToMapW(0xe5, 0xFE, 0xD3);
	addToMapW(0xe5, 0xFE, 0xD4);
	addToMapW(0xe6, 0x00, 0xB5);
	addToMapW(0xe6, 0x01, 0x60);
	addToMapW(0xe6, 0x01, 0x61);
	addToMapW(0xe6, 0x03, 0xBC);
	addToMapW(0xe6, 0x04, 0x22);
	addToMapW(0xe6, 0x04, 0x42);
	addToMapW(0xe6, 0x0E, 0x46);
	addToMapW(0xe7, 0x03, 0xA4);
	addToMapW(0xe7, 0x03, 0xC4);
	addToMapW(0xe7, 0x06, 0x42);
	addToMapW(0xe7, 0x0E, 0x47);
	addToMapW(0xe7, 0xF8, 0xBC);
	addToMapW(0xe7, 0xFE, 0xD5);
	addToMapW(0xe7, 0xFE, 0xD6);
	addToMapW(0xe7, 0xFE, 0xD7);
	addToMapW(0xe7, 0xFE, 0xD8);
	addToMapW(0xe8, 0x00, 0xD7);
	addToMapW(0xe8, 0x00, 0xDE);
	addToMapW(0xe8, 0x00, 0xFE);
	addToMapW(0xe8, 0x01, 0x36);
	addToMapW(0xe8, 0x01, 0x37);
	addToMapW(0xe8, 0x01, 0x54);
	addToMapW(0xe8, 0x01, 0x55);
	addToMapW(0xe8, 0x02, 0x78);
	addToMapW(0xe8, 0x03, 0xA6);
	addToMapW(0xe8, 0x03, 0xC6);
	addToMapW(0xe8, 0x04, 0x23);
	addToMapW(0xe8, 0x04, 0x43);
	addToMapW(0xe8, 0x06, 0x43);
	addToMapW(0xe8, 0x0E, 0x48);
	addToMapW(0xe8, 0x22, 0x05);
	addToMapW(0xe8, 0xFE, 0xD9);
	addToMapW(0xe8, 0xFE, 0xDA);
	addToMapW(0xe8, 0xFE, 0xDB);
	addToMapW(0xe8, 0xFE, 0xDC);
	addToMapW(0xe9, 0x00, 0xDA);
	addToMapW(0xe9, 0x00, 0xFA);
	addToMapW(0xe9, 0x03, 0x98);
	addToMapW(0xe9, 0x06, 0x44);
	addToMapW(0xe9, 0x0E, 0x49);
	addToMapW(0xe9, 0xFE, 0xDD);
	addToMapW(0xe9, 0xFE, 0xDE);
	addToMapW(0xe9, 0xFE, 0xDF);
	addToMapW(0xe9, 0xFE, 0xE0);
	addToMapW(0xea, 0x00, 0xDB);
	addToMapW(0xea, 0x00, 0xFB);
	addToMapW(0xea, 0x01, 0x3B);
	addToMapW(0xea, 0x01, 0x3C);
	addToMapW(0xea, 0x03, 0x86);
	addToMapW(0xea, 0x03, 0xA9);
	addToMapW(0xea, 0x03, 0xAC);
	addToMapW(0xea, 0x04, 0x16);
	addToMapW(0xea, 0x04, 0x36);
	addToMapW(0xea, 0x06, 0x45);
	addToMapW(0xea, 0x0E, 0x4A);
	addToMapW(0xea, 0x21, 0x26);
	addToMapW(0xea, 0xFE, 0xE1);
	addToMapW(0xea, 0xFE, 0xE2);
	addToMapW(0xea, 0xFE, 0xE3);
	addToMapW(0xea, 0xFE, 0xE4);
	addToMapW(0xeb, 0x00, 0xD9);
	addToMapW(0xeb, 0x00, 0xF9);
	addToMapW(0xeb, 0x01, 0x70);
	addToMapW(0xeb, 0x01, 0x71);
	addToMapW(0xeb, 0x03, 0x88);
	addToMapW(0xeb, 0x03, 0x94);
	addToMapW(0xeb, 0x03, 0xAD);
	addToMapW(0xeb, 0x03, 0xB4);
	addToMapW(0xeb, 0x06, 0x46);
	addToMapW(0xeb, 0x0E, 0x4B);
	addToMapW(0xeb, 0xFE, 0xE5);
	addToMapW(0xeb, 0xFE, 0xE6);
	addToMapW(0xeb, 0xFE, 0xE7);
	addToMapW(0xeb, 0xFE, 0xE8);
	addToMapW(0xec, 0x03, 0x01);
	addToMapW(0xec, 0x03, 0x89);
	addToMapW(0xec, 0x03, 0xAE);
	addToMapW(0xec, 0x04, 0x12);
	addToMapW(0xec, 0x04, 0x32);
	addToMapW(0xec, 0x06, 0x47);
	addToMapW(0xec, 0x0E, 0x4C);
	addToMapW(0xec, 0x22, 0x1E);
	addToMapW(0xec, 0xFE, 0xE9);
	addToMapW(0xec, 0xFE, 0xEA);
	addToMapW(0xec, 0xFE, 0xEB);
	addToMapW(0xec, 0xFE, 0xEC);
	addToMapW(0xed, 0x00, 0xDD);
	addToMapW(0xed, 0x00, 0xFD);
	addToMapW(0xed, 0x01, 0x12);
	addToMapW(0xed, 0x01, 0x13);
	addToMapW(0xed, 0x03, 0x8A);
	addToMapW(0xed, 0x03, 0xAF);
	addToMapW(0xed, 0x06, 0x48);
	addToMapW(0xed, 0x0E, 0x4D);
	addToMapW(0xed, 0xFE, 0xED);
	addToMapW(0xed, 0xFE, 0xEE);
	addToMapW(0xee, 0x00, 0xAF);
	addToMapW(0xee, 0x01, 0x45);
	addToMapW(0xee, 0x01, 0x46);
	addToMapW(0xee, 0x03, 0x04);
	addToMapW(0xee, 0x03, 0x05);
	addToMapW(0xee, 0x03, 0x8C);
	addToMapW(0xee, 0x03, 0x95);
	addToMapW(0xee, 0x03, 0xB5);
	addToMapW(0xee, 0x03, 0xCC);
	addToMapW(0xee, 0x04, 0x2C);
	addToMapW(0xee, 0x04, 0x4C);
	addToMapW(0xee, 0x06, 0x49);
	addToMapW(0xee, 0x0E, 0x4E);
	addToMapW(0xee, 0xFE, 0xEF);
	addToMapW(0xee, 0xFE, 0xF0);
	addToMapW(0xef, 0x00, 0xB4);
	addToMapW(0xef, 0x02, 0xB9);
	addToMapW(0xef, 0x02, 0xCA);
	addToMapW(0xef, 0x03, 0x01);
	addToMapW(0xef, 0x03, 0x8E);
	addToMapW(0xef, 0x03, 0xCD);
	addToMapW(0xef, 0x06, 0x4A);
	addToMapW(0xef, 0x0E, 0x4F);
	addToMapW(0xef, 0x20, 0x19);
	addToMapW(0xef, 0x20, 0x32);
	addToMapW(0xef, 0x20, 0x35);
	addToMapW(0xef, 0x21, 0x16);
	addToMapW(0xef, 0x22, 0x29);
	addToMapW(0xef, 0xFE, 0xF1);
	addToMapW(0xef, 0xFE, 0xF2);
	addToMapW(0xef, 0xFE, 0xF3);
	addToMapW(0xef, 0xFE, 0xF4);
	addToMapW(0xf0, 0x00, 0xAD);
	addToMapW(0xf0, 0x03, 0x8F);
	addToMapW(0xf0, 0x03, 0xCE);
	addToMapW(0xf0, 0x04, 0x01);
	addToMapW(0xf0, 0x04, 0x51);
	addToMapW(0xf0, 0x0E, 0x50);
	addToMapW(0xf0, 0x22, 0x61);
	addToMapW(0xf1, 0x00, 0xB1);
	addToMapW(0xf1, 0x02, 0xDD);
	addToMapW(0xf1, 0x06, 0x4B);
	addToMapW(0xf1, 0x0E, 0x51);
	addToMapW(0xf1, 0x22, 0x13);
	addToMapW(0xf1, 0xFE, 0x70);
	addToMapW(0xf1, 0xFE, 0x71);
	addToMapW(0xf2, 0x02, 0xDB);
	addToMapW(0xf2, 0x03, 0x23);
	addToMapW(0xf2, 0x04, 0x04);
	addToMapW(0xf2, 0x04, 0x2B);
	addToMapW(0xf2, 0x04, 0x4B);
	addToMapW(0xf2, 0x04, 0x54);
	addToMapW(0xf2, 0x06, 0x4C);
	addToMapW(0xf2, 0x0E, 0x52);
	addToMapW(0xf2, 0x20, 0x17);
	addToMapW(0xf2, 0x20, 0x1C);
	addToMapW(0xf2, 0x22, 0x65);
	addToMapW(0xf2, 0xF8, 0xBD);
	addToMapW(0xf2, 0xFE, 0x72);
	addToMapW(0xf3, 0x00, 0xBE);
	addToMapW(0xf3, 0x02, 0xC7);
	addToMapW(0xf3, 0x03, 0x0C);
	addToMapW(0xf3, 0x06, 0x4D);
	addToMapW(0xf3, 0x0E, 0x53);
	addToMapW(0xf3, 0x22, 0x64);
	addToMapW(0xf3, 0xFE, 0x74);
	addToMapW(0xf4, 0x00, 0xB6);
	addToMapW(0xf4, 0x02, 0xD8);
	addToMapW(0xf4, 0x03, 0x06);
	addToMapW(0xf4, 0x03, 0xAA);
	addToMapW(0xf4, 0x03, 0xCA);
	addToMapW(0xf4, 0x04, 0x07);
	addToMapW(0xf4, 0x04, 0x17);
	addToMapW(0xf4, 0x04, 0x37);
	addToMapW(0xf4, 0x04, 0x57);
	addToMapW(0xf4, 0x06, 0x4E);
	addToMapW(0xf4, 0x0E, 0x54);
	addToMapW(0xf4, 0x23, 0x20);
	addToMapW(0xf4, 0xFE, 0x76);
	addToMapW(0xf4, 0xFE, 0x77);
	addToMapW(0xf5, 0x00, 0xA7);
	addToMapW(0xf5, 0x03, 0xAB);
	addToMapW(0xf5, 0x03, 0xCB);
	addToMapW(0xf5, 0x06, 0x4F);
	addToMapW(0xf5, 0x0E, 0x55);
	addToMapW(0xf5, 0x23, 0x21);
	addToMapW(0xf5, 0xFE, 0x78);
	addToMapW(0xf5, 0xFE, 0x79);
	addToMapW(0xf6, 0x00, 0xF7);
	addToMapW(0xf6, 0x04, 0x0E);
	addToMapW(0xf6, 0x04, 0x28);
	addToMapW(0xf6, 0x04, 0x48);
	addToMapW(0xf6, 0x04, 0x5E);
	addToMapW(0xf6, 0x06, 0x50);
	addToMapW(0xf6, 0x0E, 0x56);
	addToMapW(0xf6, 0xFE, 0x7A);
	addToMapW(0xf6, 0xFE, 0x7B);
	addToMapW(0xf7, 0x00, 0xB8);
	addToMapW(0xf7, 0x00, 0xF7);
	addToMapW(0xf7, 0x02, 0xDB);
	addToMapW(0xf7, 0x03, 0x27);
	addToMapW(0xf7, 0x0E, 0x57);
	addToMapW(0xf7, 0x20, 0x1E);
	addToMapW(0xf7, 0x22, 0x48);
	addToMapW(0xf8, 0x00, 0xB0);
	addToMapW(0xf8, 0x02, 0xDA);
	addToMapW(0xf8, 0x03, 0x0A);
	addToMapW(0xf8, 0x04, 0x2D);
	addToMapW(0xf8, 0x04, 0x4D);
	addToMapW(0xf8, 0x0E, 0x58);
	addToMapW(0xf8, 0x20, 0x70);
	addToMapW(0xf8, 0x22, 0x18);
	addToMapW(0xf9, 0x00, 0xA8);
	addToMapW(0xf9, 0x02, 0xDD);
	addToMapW(0xf9, 0x03, 0x08);
	addToMapW(0xf9, 0x0E, 0x59);
	addToMapW(0xf9, 0x22, 0x19);
	addToMapW(0xfa, 0x00, 0xB7);
	addToMapW(0xfa, 0x02, 0xD9);
	addToMapW(0xfa, 0x03, 0x07);
	addToMapW(0xfa, 0x04, 0x29);
	addToMapW(0xfa, 0x04, 0x49);
	addToMapW(0xfa, 0x0E, 0x5A);
	addToMapW(0xfa, 0x20, 0x24);
	addToMapW(0xfa, 0x22, 0xC5);
	addToMapW(0xfa, 0x30, 0xFB);
	addToMapW(0xfb, 0x00, 0xB9);
	addToMapW(0xfb, 0x0E, 0x5B);
	addToMapW(0xfb, 0x20, 0x81);
	addToMapW(0xfb, 0x22, 0x1A);
	addToMapW(0xfb, 0x27, 0x13);
	addToMapW(0xfc, 0x00, 0xB3);
	addToMapW(0xfc, 0x01, 0x58);
	addToMapW(0xfc, 0x01, 0x59);
	addToMapW(0xfc, 0x04, 0x27);
	addToMapW(0xfc, 0x04, 0x47);
	addToMapW(0xfc, 0x20, 0x7F);
	addToMapW(0xfc, 0x20, 0x83);
	addToMapW(0xfc, 0x21, 0x16);
	addToMapW(0xfc, 0xF8, 0xC5);
	addToMapW(0xfd, 0x00, 0xA4);
	addToMapW(0xfd, 0x00, 0xA7);
	addToMapW(0xfd, 0x00, 0xB2);
	addToMapW(0xfd, 0x20, 0x82);
	addToMapW(0xfd, 0xF8, 0xC6);
	addToMapW(0xfd, 0xF8, 0xF1);
	addToMapW(0xfe, 0x20, 0xAB);
	addToMapW(0xfe, 0x25, 0xA0);
	addToMapW(0xfe, 0xF8, 0xC7);
	addToMapW(0xfe, 0xF8, 0xF2);
	addToMapW(0xff, 0x00, 0xA0);
	addToMapW(0xff, 0xF8, 0xC8);
	addToMapW(0xff, 0xF8, 0xF3);
}
