#!/usr/bin/env python

import pigpio

class hasher:

   """
   This class forms a hash over the IR pulses generated by an
   IR remote.

   The remote key press is not converted into a code in the manner of
   the lirc module.  No attempt is made to decode the type of protocol
   used by the remote.  The hash is likely to be unique for different
   keys and different remotes but this is not guaranteed.

   This hashing process works for some remotes/protocols but not for
   others.  The only way to find out if it works for one or more of
   your remotes is to try it and see.

   EXAMPLE CODE

   #!/usr/bin/env python

   import time
   import pigpio
   import ir_hasher

   def callback(hash):
      print("hash={}".format(hash));

   pi = pigpio.pi()

   ir = ir_hasher.hasher(pi, 7, callback, 5)

   print("ctrl c to exit");

   time.sleep(300)

   pi.stop()
   """

   def __init__(self, pi, gpio, callback, timeout=5):

      """
      Initialises an IR remote hasher on a pi's gpio.  A gap of timeout
      milliseconds indicates the end of the remote key press.
      """

      self.pi = pi
      self.gpio = gpio
      self.code_timeout = timeout
      self.callback = callback

      self.in_code = False

      pi.set_mode(gpio, pigpio.INPUT)

      self.cb = pi.callback(gpio, pigpio.EITHER_EDGE, self._cb)

   def _hash(self, old_val, new_val):

      if   new_val < (old_val * 0.60):
         val = 13
      elif old_val < (new_val * 0.60):
         val = 23
      else:
         val = 2

      self.hash_val = self.hash_val ^ val
      self.hash_val *= 16777619 # FNV_PRIME_32
      self.hash_val = self.hash_val & ((1<<32)-1)

   def _cb(self, gpio, level, tick):

      if level != pigpio.TIMEOUT:

         if self.in_code == False:

            self.in_code = True

            self.pi.set_watchdog(self.gpio, self.code_timeout)

            self.hash_val = 2166136261 # FNV_BASIS_32

            self.edges = 1

            self.t1 = None
            self.t2 = None
            self.t3 = None
            self.t4 = tick

         else:

            self.edges += 1

            self.t1 = self.t2
            self.t2 = self.t3
            self.t3 = self.t4
            self.t4 = tick

            if self.t1 is not None:

               d1 = pigpio.tickDiff(self.t1,self.t2)
               d2 = pigpio.tickDiff(self.t3,self.t4)

               self._hash(d1, d2)

      else:

         if self.in_code:

            self.in_code = False

            self.pi.set_watchdog(self.gpio, 0)

            if self.edges > 12:

               self.callback(self.hash_val)

if __name__ == "__main__":

   import time
   import pigpio
   import ir_hasher

   hashes = {
      142650387: '2',       244341844: 'menu',    262513468: 'vol-',
      272048826: '5',       345069212: '6',       363685443: 'prev.ch',
      434191356: '1',       492745084: 'OK',      549497027: 'mute',
      603729091: 'text',    646476378: 'chan-',   832916949: 'home',
      923778138: 'power',   938165610: 'power',   953243510: 'forward',
      1009731980:'1',       1018231875:'TV',      1142888517:'c-up',
      1151589683:'chan+',   1344018636:'OK',      1348032067:'chan+',
      1367109971:'prev.ch', 1370712102:'c-left',  1438405361:'rewind',
      1452589043:'pause',   1518578730:'chan-',   1554432645:'8',
      1583569525:'0',       1629745313:'rewind',  1666513749:'record',
      1677653754:'c-down',  1825951717:'c-right', 1852412236:'6',
      1894279468:'9',       1904895749:'vol+',    1941947509:'ff',
      2076573637:'0',       2104823531:'back',    2141641957:'home',
      2160787557:'record',  2398525299:'7',       2468117013:'8',
      2476712746:'play',    2574308838:'forward', 2577952149:'4',
      2706654902:'stop',    2829002741:'c-up',    2956097083:'back',
      3112717386:'5',       3263244773:'ff',      3286088195:'pause',
      3363767978:'c-down',  3468076364:'vol-',    3491068358:'stop',
      3593710134:'c-left',  3708232515:'3',       3734134565:'back',
      3766109107:'TV',      3798010010:'play',    3869937700:'menu',
      3872715523:'7',       3885097091:'2',       3895301587:'text',
      3931058739:'mute',    3983900853:'c-right', 4032250885:'4',
      4041913909:'vol+',    4207017660:'9',       4227138677:'back',
      4294027955:'3'}

   def callback(hash):
      if hash in hashes:
         print("key={} hash={}".format(hashes[hash], hash));

   pi = pigpio.pi()

   ir = ir_hasher.hasher(pi, 7, callback, 5)

   print("ctrl c to exit");

   time.sleep(300)

   pi.stop()

